package com.siguiente.fasalexpert.utils

import android.app.Activity
import android.content.Context
import android.content.Intent
import android.graphics.*
import android.os.Build
import android.view.View
import android.widget.*
import androidx.annotation.RequiresApi
import androidx.core.content.ContextCompat
import androidx.databinding.BindingAdapter
import com.bumptech.glide.Glide
import com.google.android.material.snackbar.Snackbar
import com.google.android.material.switchmaterial.SwitchMaterial
import com.google.android.material.textfield.TextInputLayout
import java.text.SimpleDateFormat
import java.util.*



fun Context.toast(message: String) {
    Toast.makeText(this, message, Toast.LENGTH_LONG).show()
}

@BindingAdapter("app:errorText")
fun setErrorMessage(view: TextInputLayout, errorMessage: String) {
    view.error = errorMessage
}

@BindingAdapter("thumb")
fun loadImage(view: ImageView, url: String) {
    Glide.with(view)
        .load(url)
        .into(view)
}

@BindingAdapter("checked")
fun isActiveSwitch(view: SwitchMaterial, status: Int) {
    view.isChecked = status == 1
}



@BindingAdapter("bindFormattedDate")
fun bindFormattedDate(textView: TextView, dateInput: String?) {
    val fromServer = SimpleDateFormat("yyyy-MM-dd")
    val myFormat = SimpleDateFormat("EEE, MMM d, ''yy")
    val date: Date = fromServer.parse(dateInput)
    val outputDateStr: String = myFormat.format(date)
    textView.text = outputDateStr
}


fun Context.fetchColor(id: Int): Int = ContextCompat.getColor(this, id)

fun <A : Activity> Activity.startNewActivity(activity: Class<A>) {
    Intent(this, activity).also {
        it.flags = Intent.FLAG_ACTIVITY_NEW_TASK or Intent.FLAG_ACTIVITY_CLEAR_TASK
        startActivity(it)
    }
}

fun View.visible(isVisible: Boolean) {
    visibility = if (isVisible) View.VISIBLE else View.GONE
}

fun View.enable(enabled: Boolean) {
    isEnabled = enabled
    alpha = if (enabled) 1f else 0.5f
}

fun View.snackbar(message: String, action: (() -> Unit)? = null) {
    val snackbar = Snackbar.make(this, message, Snackbar.LENGTH_LONG)
    action?.let {
        snackbar.setAction("Retry") {
            it()
        }
    }
    snackbar.show()
}

fun getCircularBitmap(srcBitmap: Bitmap): Bitmap? {
    val squareBitmapWidth = Math.min(srcBitmap.width, srcBitmap.height)
    val dstBitmap = Bitmap.createBitmap(
        squareBitmapWidth,  // Width
        squareBitmapWidth,  // Height
        Bitmap.Config.ARGB_8888 // Config
    )
    val canvas = Canvas(dstBitmap)
    val paint = Paint()
    paint.isAntiAlias = true
    val rect = Rect(0, 0, squareBitmapWidth, squareBitmapWidth)
    val rectF = RectF(rect)
    canvas.drawOval(rectF, paint)
    paint.xfermode = PorterDuffXfermode(PorterDuff.Mode.SRC_IN)
    val left = ((squareBitmapWidth - srcBitmap.width) / 2).toFloat()
    val top = ((squareBitmapWidth - srcBitmap.height) / 2).toFloat()
    canvas.drawBitmap(srcBitmap, left, top, paint)
    srcBitmap.recycle()
    return dstBitmap
}

@RequiresApi(Build.VERSION_CODES.M)
fun createImage(width: Int, height: Int, colorBG: Int, colorText: Int, name: String?): Bitmap? {
    val bitmap = Bitmap.createBitmap(width, height, Bitmap.Config.ARGB_8888)
    val canvas = Canvas(bitmap)
    val paint2 = Paint()
    paint2.setColor(colorBG)
    canvas.drawRect(0f, 0f, width.toFloat(), height.toFloat(), paint2)
    val paint = Paint()
    paint.setColor(colorText)
    paint.setTextSize(50f)
    paint.setTextScaleX(1f)
    canvas.drawText(name!!, 50f - 15, 50f + 15, paint)
    return getCircularBitmap(bitmap)
}


fun ProgressBar.show() {
    visibility = View.VISIBLE
}

fun ProgressBar.hide() {
    visibility = View.GONE
}

fun View.snackbar(message: String) {
    Snackbar.make(this, message, Snackbar.LENGTH_LONG).also { snackbar ->
        snackbar.setAction("Ok") {
            snackbar.dismiss()
        }
    }.show()
}
