package com.siguiente.fasal.imageCrop

import android.animation.ValueAnimator
import android.animation.ValueAnimator.AnimatorUpdateListener
import android.annotation.SuppressLint
import android.annotation.TargetApi
import android.content.Context
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.graphics.Canvas
import android.graphics.RectF
import android.graphics.drawable.Drawable
import android.net.Uri
import android.os.AsyncTask
import android.os.Build
import android.util.AttributeSet
import android.view.GestureDetector
import android.view.MotionEvent
import android.view.ScaleGestureDetector
import android.view.ScaleGestureDetector.OnScaleGestureListener
import android.view.View
import android.view.animation.DecelerateInterpolator
import androidx.annotation.RequiresApi
import kotlin.math.abs
import kotlin.math.sqrt


class InstaCropperView : View {

    interface BitmapCallback {
        fun onBitmapReady(bitmap: Bitmap?)
    }

    private var mMinimumRatio = DEFAULT_MINIMUM_RATIO
    private var mMaximumRatio = DEFAULT_MAXIMUM_RATIO
    private var mDefaultRatio = DEFAULT_RATIO
    private var mImageUri: Uri? = null
    private var mImageRawWidth = 0
    private var mImageRawHeight = 0
    private var mMakeDrawableTask: MakeDrawableTask? = null
    private var mWidth = 0
    private var mHeight = 0
    private val mGridDrawable = GridDrawable()
    private var mDrawable: Drawable? = null
    private var mDrawableScale = 0f
    private var mScaleFocusX = 0f
    private var mScaleFocusY = 0f
    private var mDisplayDrawableLeft = 0f
    private var mDisplayDrawableTop = 0f
    private val mHelperRect = RectF()
    private var mGestureDetector: GestureDetector? = null
    private var mScaleGestureDetector: ScaleGestureDetector? = null
    private var mMaximumOverScroll = 0f
    private var mAnimator: ValueAnimator? = null
    private var mOnClickListener: OnClickListener? = null

    constructor(context: Context) : super(context) {
        initialize(context, null, 0, 0)
    }

    constructor(context: Context, attrs: AttributeSet?) : super(context, attrs) {
        initialize(context, attrs, 0, 0)
    }

    constructor(context: Context, attrs: AttributeSet?, defStyleAttr: Int) : super(
        context,
        attrs,
        defStyleAttr
    ) {
        initialize(context, attrs, defStyleAttr, 0)
    }

    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    constructor(
        context: Context,
        attrs: AttributeSet?,
        defStyleAttr: Int,
        defStyleRes: Int
    ) : super(context, attrs, defStyleAttr, defStyleRes) {
        initialize(context, attrs, defStyleAttr, defStyleRes)
    }

    private fun initialize(
        context: Context,
        attrs: AttributeSet?,
        defStyleAttr: Int,
        defStyleRes: Int
    ) {
        mGestureDetector = GestureDetector(context, mOnGestureListener)
        mScaleGestureDetector = ScaleGestureDetector(context, mOnScaleGestureListener)
        mMaximumOverScroll = resources.displayMetrics.density * MAXIMUM_OVER_SCROLL
        mAnimator = ValueAnimator()
        mAnimator!!.duration = SET_BACK_DURATION
        mAnimator!!.setFloatValues(0f, 1f)
        mAnimator!!.interpolator = DecelerateInterpolator(0.25f)
        mAnimator!!.addUpdateListener(mSettleAnimatorUpdateListener)
        mGridDrawable.callback = mGridCallback
    }

    private val mGridCallback: Drawable.Callback = object : Drawable.Callback {
        override fun invalidateDrawable(who: Drawable) {
            invalidate()
        }

        override fun scheduleDrawable(who: Drawable, what: Runnable, `when`: Long) {}
        override fun unscheduleDrawable(who: Drawable, what: Runnable) {}
    }

    fun setRatios(defaultRatio: Float, minimumRatio: Float, maximumRatio: Float) {
        mDefaultRatio = defaultRatio
        mMinimumRatio = minimumRatio
        mMaximumRatio = maximumRatio
        if (mAnimator!!.isRunning) {
            mAnimator!!.cancel()
        }
        cancelMakingDrawableProcessIfExists()
        mDrawable = null
        requestLayout()
    }

    fun setImageUri(uri: Uri?) {
        cancelMakingDrawableProcessIfExists()
        mImageUri = uri
        mDrawable = null
        requestLayout()
        invalidate()
    }

    @SuppressLint("StaticFieldLeak")
    fun crop(widthSpec: Int, heightSpec: Int, callback: BitmapCallback) {
        checkNotNull(mImageUri) { "Image uri is not set." }
        if (mDrawable == null || mAnimator!!.isRunning) {
            postDelayed({ crop(widthSpec, heightSpec, callback) }, SET_BACK_DURATION / 2)
            return
        }
        val gridBounds = RectF(mGridDrawable.bounds)
        gridBounds.offset(-mDisplayDrawableLeft, -mDisplayDrawableTop)
        getDisplayDrawableBounds(mHelperRect)
        val leftRatio = gridBounds.left / mHelperRect.width()
        val topRatio = gridBounds.top / mHelperRect.height()
        val rightRatio = gridBounds.right / mHelperRect.width()
        val bottomRatio = gridBounds.bottom / mHelperRect.height()
        val actualLeft = 0.coerceAtLeast((leftRatio * mImageRawWidth).toInt())
        val actualTop = 0.coerceAtLeast((topRatio * mImageRawHeight).toInt())
        val actualRight = mImageRawWidth.coerceAtMost((rightRatio * mImageRawWidth).toInt())
        val actualBottom = mImageRawHeight.coerceAtMost((bottomRatio * mImageRawHeight).toInt())
        val context = context


        object : AsyncTask<Void?, Void?, Bitmap?>() {
            @RequiresApi(Build.VERSION_CODES.Q)
            @SuppressLint("StaticFieldLeak")

            protected override fun doInBackground(vararg params: Void?): Bitmap? {
                val actualWidth = actualRight - actualLeft
                val actualHeight = actualBottom - actualTop
                var actualRatio = actualWidth.toFloat() / actualHeight.toFloat()
                if (actualRatio < mMinimumRatio) {
                    actualRatio = mMinimumRatio
                }
                if (actualRatio > mMaximumRatio) {
                    actualRatio = mMaximumRatio
                }
                val widthMode = MeasureSpec.getMode(widthSpec)
                val widthSize = MeasureSpec.getSize(widthSpec)
                val heightMode = MeasureSpec.getMode(heightSpec)
                val heightSize = MeasureSpec.getSize(heightSpec)
                var targetWidth = actualWidth
                var targetHeight = actualHeight
                when (widthMode) {
                    MeasureSpec.EXACTLY -> {
                        targetWidth = widthSize
                        when (heightMode) {
                            MeasureSpec.EXACTLY -> targetHeight = heightSize
                            MeasureSpec.AT_MOST -> targetHeight =
                                heightSize.coerceAtMost((targetWidth / actualRatio).toInt())
                            MeasureSpec.UNSPECIFIED -> targetHeight =
                                (targetWidth / actualRatio).toInt()
                        }
                    }
                    MeasureSpec.AT_MOST -> when (heightMode) {
                        MeasureSpec.EXACTLY -> {
                            targetHeight = heightSize
                            targetWidth =
                                widthSize.coerceAtMost((targetHeight * actualRatio).toInt())
                        }
                        MeasureSpec.AT_MOST -> if (actualWidth <= widthSize && actualHeight <= heightSize) {
                            targetWidth = actualWidth
                            targetHeight = actualHeight
                        } else {
                            val specRatio = widthSize.toFloat() / heightSize.toFloat()
                            when {
                                specRatio == actualRatio -> {
                                    targetWidth = widthSize
                                    targetHeight = heightSize
                                }
                                specRatio > actualRatio -> {
                                    targetHeight = heightSize
                                    targetWidth = (targetHeight * actualRatio).toInt()
                                }
                                else -> {
                                    targetWidth = widthSize
                                    targetHeight = (targetWidth / actualRatio).toInt()
                                }
                            }
                        }
                        MeasureSpec.UNSPECIFIED -> if (actualWidth <= widthSize) {
                            targetWidth = actualWidth
                            targetHeight = actualHeight
                        } else {
                            targetWidth = widthSize
                            targetHeight = (targetWidth / actualRatio).toInt()
                        }
                    }
                    MeasureSpec.UNSPECIFIED -> when (heightMode) {
                        MeasureSpec.EXACTLY -> {
                            targetHeight = heightSize
                            targetWidth = (targetHeight * actualRatio).toInt()
                        }
                        MeasureSpec.AT_MOST -> if (actualHeight <= heightSize) {
                            targetHeight = actualHeight
                            targetWidth = actualWidth
                        } else {
                            targetHeight = heightSize
                            targetWidth = (targetHeight * actualRatio).toInt()
                        }
                        MeasureSpec.UNSPECIFIED -> {
                            targetWidth = actualWidth
                            targetHeight = actualHeight
                        }
                    }
                }
                return cropImageAndResize(
                    context,
                    actualLeft,
                    actualTop,
                    actualRight,
                    actualBottom,
                    targetWidth,
                    targetHeight
                )
            }

            override fun onPostExecute(bitmap: Bitmap?) {
                callback.onBitmapReady(bitmap)
            }
        }.execute()
    }

    @RequiresApi(Build.VERSION_CODES.Q)
    private fun cropImageAndResize(
        context: Context,
        left: Int,
        top: Int,
        right: Int,
        bottom: Int,
        width: Int,
        height: Int
    ): Bitmap? {
        var left = left
        var top = top
        var right = right
        var bottom = bottom
        val options = BitmapFactory.Options()
        options.inSampleSize = 1
        val rawArea = (right - left) * (bottom - top)
        val targetArea = width * height * 4
        var resultArea = rawArea
        while (resultArea > targetArea) {
            options.inSampleSize *= 2
            resultArea = rawArea / (options.inSampleSize * options.inSampleSize)
        }
        if (options.inSampleSize > 1) {
            options.inSampleSize /= 2
        }
        return try {
            val rawBitmap: Bitmap = MakeDrawableTask.getBitmap(context, mImageUri!!, options)
                ?: return null
            left /= options.inSampleSize
            top /= options.inSampleSize
            right /= options.inSampleSize
            bottom /= options.inSampleSize
            val croppedWidth = right - left
            val croppedHeight = bottom - top
            val croppedBitmap =
                Bitmap.createBitmap(rawBitmap, left, top, croppedWidth, croppedHeight)
            if (rawBitmap != croppedBitmap) {
                rawBitmap.recycle()
            }
            if (croppedWidth <= width && croppedHeight <= height) {
                return croppedBitmap
            }
            val resizedBitmap: Bitmap = MakeDrawableTask.resizeBitmap(croppedBitmap, width, height)
            croppedBitmap.recycle()
            resizedBitmap
        } catch (t: Throwable) {
            null
        }
    }

    override fun onMeasure(widthMeasureSpec: Int, heightMeasureSpec: Int) {
        val widthMode = MeasureSpec.getMode(widthMeasureSpec)
        val widthSize = MeasureSpec.getSize(widthMeasureSpec)
        val heightMode = MeasureSpec.getMode(heightMeasureSpec)
        val heightSize = MeasureSpec.getSize(heightMeasureSpec)
        var targetWidth = 1
        var targetHeight = 1
        when (widthMode) {
            MeasureSpec.EXACTLY -> {
                targetWidth = widthSize
                when (heightMode) {
                    MeasureSpec.EXACTLY -> targetHeight = heightSize
                    MeasureSpec.AT_MOST -> targetHeight =
                        heightSize.coerceAtMost((targetWidth / mDefaultRatio).toInt())
                    MeasureSpec.UNSPECIFIED -> targetHeight = (targetWidth / mDefaultRatio).toInt()
                }
            }
            MeasureSpec.AT_MOST -> when (heightMode) {
                MeasureSpec.EXACTLY -> {
                    targetHeight = heightSize
                    targetWidth = widthSize.coerceAtMost((targetHeight * mDefaultRatio).toInt())
                }
                MeasureSpec.AT_MOST -> {
                    val specRatio = widthSize.toFloat() / heightSize.toFloat()
                    if (specRatio == mDefaultRatio) {
                        targetWidth = widthSize
                        targetHeight = heightSize
                    } else if (specRatio > mDefaultRatio) {
                        targetHeight = heightSize
                        targetWidth = (targetHeight * mDefaultRatio).toInt()
                    } else {
                        targetWidth = widthSize
                        targetHeight = (targetWidth / mDefaultRatio).toInt()
                    }
                }
                MeasureSpec.UNSPECIFIED -> {
                    targetWidth = widthSize
                    targetHeight = (targetWidth / mDefaultRatio).toInt()
                }
            }
            MeasureSpec.UNSPECIFIED -> when (heightMode) {
                MeasureSpec.EXACTLY -> {
                    targetHeight = heightSize
                    targetWidth = (targetHeight * mDefaultRatio).toInt()
                }
                MeasureSpec.AT_MOST -> {
                    targetHeight = heightSize
                    targetWidth = (targetHeight * mDefaultRatio).toInt()
                }
                MeasureSpec.UNSPECIFIED -> {
                    targetWidth = mMaximumOverScroll.toInt()
                    targetHeight = mMaximumOverScroll.toInt()
                }
            }
        }
        setMeasuredDimension(targetWidth, targetHeight)
    }

    override fun onLayout(changed: Boolean, left: Int, top: Int, right: Int, bottom: Int) {
        super.onLayout(changed, left, top, right, bottom)
        mWidth = right - left
        mHeight = bottom - top
        if (mWidth == 0 || mHeight == 0) {
            return
        }
        if (mImageUri == null) {
            return
        }
        if (currentDrawableIsSuitableForView()) {
            cancelMakingDrawableProcessIfExists()
            return
        }
        if (isMakingDrawableForView) {
            if (drawableBeingMadeIsSuitableForView()) {
                return
            }
            cancelMakingDrawableProcessIfExists()
        }
        startMakingSuitableDrawable()
    }

    private fun currentDrawableIsSuitableForView(): Boolean {
        if (mDrawable == null) {
            return false
        }
        val drawableWidth = mDrawable!!.intrinsicWidth
        val drawableHeight = mDrawable!!.intrinsicHeight
        return isSizeSuitableForView(drawableWidth, drawableHeight)
    }

    private fun cancelMakingDrawableProcessIfExists() {
        if (mMakeDrawableTask != null) {
            mMakeDrawableTask!!.cancel(true)
            mMakeDrawableTask = null
        }
    }

    private val isMakingDrawableForView: Boolean
        get() = mMakeDrawableTask != null

    private fun drawableBeingMadeIsSuitableForView(): Boolean {
        return isSizeSuitableForView(
            mMakeDrawableTask!!.targetWidth,
            mMakeDrawableTask!!.targetHeight
        )
    }

    private fun isSizeSuitableForView(width: Int, height: Int): Boolean {
        val viewArea = mWidth * mHeight
        val drawableArea = width * height
        val areaRatio = viewArea.toFloat() / drawableArea.toFloat()
        return areaRatio in 0.5f..2f
    }

    private fun startMakingSuitableDrawable() {
        mMakeDrawableTask = @SuppressLint("StaticFieldLeak")
        object : MakeDrawableTask(context, mImageUri!!, mWidth, mHeight) {
            protected override fun onPostExecute(drawable: Drawable?) {
                mDrawable = drawable
                mImageRawWidth = rawWidth
                mImageRawHeight = rawHeight
                onDrawableChanged()
            }
        }
        (mMakeDrawableTask as MakeDrawableTask).execute()
    }

    private fun onDrawableChanged() {
        reset()
    }

    private fun reset() {
        if (mAnimator!!.isRunning) {
            mAnimator!!.cancel()
        }
        scaleDrawableToFitWithinViewWithValidRatio()
        placeDrawableInTheCenter()
        updateGrid()
        invalidate()
    }

    private fun isImageSizeRatioValid(imageSizeRatio: Float): Boolean {
        return imageSizeRatio in mMinimumRatio..mMaximumRatio
    }

    private val imageSizeRatio: Float
        get() = mImageRawWidth.toFloat() / mImageRawHeight.toFloat()

    private fun scaleDrawableToFitWithinViewWithValidRatio() {
        val scale = drawableScaleToFitWithValidRatio
        setDrawableScale(scale)
    }

    private val drawableScaleToFitWithValidRatio: Float
        get() {
            val scale: Float
            val drawableSizeRatio = imageSizeRatio
            val imageSizeRatioIsValid = isImageSizeRatioValid(drawableSizeRatio)
            scale = if (imageSizeRatioIsValid) {
                val viewRatio = mWidth.toFloat() / mHeight.toFloat()
                val drawableRatio = mImageRawWidth.toFloat() / mImageRawHeight.toFloat()
                val drawableIsWiderThanView = drawableRatio > viewRatio
                if (drawableIsWiderThanView) {
                    mWidth.toFloat() / mImageRawWidth.toFloat()
                } else {
                    mHeight.toFloat() / mImageRawHeight.toFloat()
                }
            } else {
                if (drawableSizeRatio < mMinimumRatio) {
                    getBoundsForHeightAndRatio(mHeight.toFloat(), mMinimumRatio, mHelperRect)
                    mHelperRect.width() / mImageRawWidth
                } else {
                    getBoundsForWidthAndRatio(mWidth.toFloat(), mMaximumRatio, mHelperRect)
                    mHelperRect.height() / mImageRawHeight
                }
            }
            return scale
        }

    private fun setDrawableScale(scale: Float) {
        mDrawableScale = scale
        invalidate()
    }

    private fun placeDrawableInTheCenter() {
        mDisplayDrawableLeft = (mWidth - displayDrawableWidth) / 2
        mDisplayDrawableTop = (mHeight - displayDrawableHeight) / 2
        invalidate()
    }

    private val displayDrawableWidth: Float
        get() = mDrawableScale * mImageRawWidth
    private val displayDrawableHeight: Float
        get() = mDrawableScale * mImageRawHeight

    private fun updateGrid() {
        getDisplayDrawableBounds(mHelperRect)
        mHelperRect.intersect(0f, 0f, mWidth.toFloat(), mHeight.toFloat())
        val gridLeft = mHelperRect.left
        val gridTop = mHelperRect.top
        val gridWidth = mHelperRect.width()
        val gridHeight = mHelperRect.height()
        mHelperRect[gridLeft, gridTop, gridLeft + gridWidth] = gridTop + gridHeight
        setGridBounds(mHelperRect)
        invalidate()
    }

    private fun getBoundsForWidthAndRatio(width: Float, ratio: Float, rect: RectF) {
        val height = width / ratio
        rect[0f, 0f, width] = height
    }

    private fun getBoundsForHeightAndRatio(height: Float, ratio: Float, rect: RectF) {
        val width = height * ratio
        rect[0f, 0f, width] = height
    }

    private fun setGridBounds(bounds: RectF) {
        mGridDrawable.setBounds(
            bounds.left.toInt(), bounds.top.toInt(),
            bounds.right.toInt(), bounds.bottom.toInt()
        )
        invalidate()
    }

    private fun getDisplayDrawableBounds(bounds: RectF) {
        bounds.left = mDisplayDrawableLeft
        bounds.top = mDisplayDrawableTop
        bounds.right = bounds.left + displayDrawableWidth
        bounds.bottom = bounds.top + displayDrawableHeight
    }

    override fun onDraw(canvas: Canvas) {
        super.onDraw(canvas)
        if (mDrawable == null) {
            return
        }
        getDisplayDrawableBounds(mHelperRect)
        mDrawable!!.setBounds(
            mHelperRect.left.toInt(),
            mHelperRect.top.toInt(), mHelperRect.right.toInt(), mHelperRect.bottom.toInt()
        )
        mDrawable!!.draw(canvas)
        mGridDrawable.draw(canvas)
    }

    override fun setOnClickListener(l: OnClickListener?) {
        mOnClickListener = l
    }

    @SuppressLint("ClickableViewAccessibility")
    override fun onTouchEvent(event: MotionEvent): Boolean {
        if (mDrawable == null) {
            return false
        }
        mGestureDetector!!.onTouchEvent(event)
        mScaleGestureDetector!!.onTouchEvent(event)
        val action = event.action
        if (action == MotionEvent.ACTION_UP || action == MotionEvent.ACTION_CANCEL || action == MotionEvent.ACTION_OUTSIDE) {
            mAnimator!!.start()
        }
        return true
    }

    private val mOnGestureListener: GestureDetector.OnGestureListener =
        object : GestureDetector.OnGestureListener {
            override fun onDown(motionEvent: MotionEvent): Boolean {
                return true
            }

            override fun onShowPress(motionEvent: MotionEvent) {}
            override fun onSingleTapUp(motionEvent: MotionEvent): Boolean {
                if (mOnClickListener != null) {
                    mOnClickListener!!.onClick(this@InstaCropperView)
                    return true
                }
                return false
            }

            override fun onScroll(
                e1: MotionEvent,
                e2: MotionEvent,
                distanceX: Float,
                distanceY: Float
            ): Boolean {
                var distanceX = distanceX
                var distanceY = distanceY
                distanceX = -distanceX
                distanceY = -distanceY
                getDisplayDrawableBounds(mHelperRect)
                val overScrollX = measureOverScrollX(mHelperRect)
                val overScrollY = measureOverScrollY(mHelperRect)
                distanceX = applyOverScrollFix(distanceX, overScrollX)
                distanceY = applyOverScrollFix(distanceY, overScrollY)
                mDisplayDrawableLeft += distanceX
                mDisplayDrawableTop += distanceY
                updateGrid()
                invalidate()
                return true
            }

            override fun onLongPress(motionEvent: MotionEvent) {}
            override fun onFling(
                motionEvent: MotionEvent,
                motionEvent1: MotionEvent,
                v: Float,
                v1: Float
            ): Boolean {
                return false
            }
        }

    private fun measureOverScrollX(displayDrawableBounds: RectF): Float {
        val drawableIsSmallerThanView = displayDrawableBounds.width() <= mWidth
        if (drawableIsSmallerThanView) {
            return displayDrawableBounds.centerX() - mWidth / 2
        }
        if (displayDrawableBounds.left <= 0 && displayDrawableBounds.right >= mWidth) {
            return 0f
        }
        if (displayDrawableBounds.left < 0) {
            return displayDrawableBounds.right - mWidth
        }
        return if (displayDrawableBounds.right > mWidth) {
            displayDrawableBounds.left
        } else 0f
    }

    private fun measureOverScrollY(displayDrawableBounds: RectF): Float {
        val drawableIsSmallerThanView = displayDrawableBounds.height() < mHeight
        if (drawableIsSmallerThanView) {
            return displayDrawableBounds.centerY() - mHeight / 2
        }
        if (displayDrawableBounds.top <= 0 && displayDrawableBounds.bottom >= mHeight) {
            return 0f
        }
        if (displayDrawableBounds.top < 0) {
            return displayDrawableBounds.bottom - mHeight
        }
        return if (displayDrawableBounds.bottom > mHeight) {
            displayDrawableBounds.top
        } else 0f
    }

    private fun applyOverScrollFix(distance: Float, overScroll: Float): Float {
        var distance = distance
        if (overScroll * distance <= 0) {
            return distance
        }
        val offRatio = abs(overScroll) / mMaximumOverScroll
        distance -= (distance * sqrt(offRatio.toDouble())).toFloat()
        return distance
    }

    private val mOnScaleGestureListener: OnScaleGestureListener = object : OnScaleGestureListener {
        override fun onScale(detector: ScaleGestureDetector): Boolean {
            val overScale = measureOverScale()
            val scale = applyOverScaleFix(detector.scaleFactor, overScale)
            mScaleFocusX = detector.focusX
            mScaleFocusY = detector.focusY
            setScaleKeepingFocus(mDrawableScale * scale, mScaleFocusX, mScaleFocusY)
            return true
        }

        override fun onScaleBegin(detector: ScaleGestureDetector): Boolean {
            return true
        }

        override fun onScaleEnd(detector: ScaleGestureDetector) {}
    }

    private fun measureOverScale(): Float {
        var maximumAllowedScale = maximumAllowedScale
        val minimumAllowedScale = minimumAllowedScale
        if (maximumAllowedScale < minimumAllowedScale) {
            maximumAllowedScale = minimumAllowedScale
        }
        return when {
            mDrawableScale < minimumAllowedScale -> {
                mDrawableScale / minimumAllowedScale
            }
            mDrawableScale > maximumAllowedScale -> {
                mDrawableScale / maximumAllowedScale
            }
            else -> {
                1f
            }
        }
    }

    private val maximumAllowedScale: Float
        get() {
            val maximumAllowedWidth = mImageRawWidth.toFloat()
            val maximumAllowedHeight = mImageRawHeight.toFloat()
            return (maximumAllowedWidth / mWidth.toFloat()).coerceAtMost(maximumAllowedHeight / mHeight.toFloat())
        }
    private val minimumAllowedScale: Float
        get() = drawableScaleToFitWithValidRatio

    private fun applyOverScaleFix(scale: Float, overScale: Float): Float {
        var scale = scale
        var overScale = overScale
        if (overScale == 1f) {
            return scale
        }
        if (overScale > 1) {
            class InstaCropperView {
            }
            overScale = 1f / overScale
        }
        var wentOverScaleRatio = (overScale - MAXIMUM_OVER_SCALE) / (1 - MAXIMUM_OVER_SCALE)
        if (wentOverScaleRatio < 0) {
            wentOverScaleRatio = 0f
        }

        // 1 -> scale , 0 -> 1
        // scale * f(1) = scale
        // scale * f(0) = 1

        // f(1) = 1
        // f(0) = 1/scale
        scale *= wentOverScaleRatio + (1 - wentOverScaleRatio) / scale
        return scale
    }

    private fun setScaleKeepingFocus(scale: Float, focusX: Float, focusY: Float) {
        getDisplayDrawableBounds(mHelperRect)
        val focusRatioX = (focusX - mHelperRect.left) / mHelperRect.width()
        val focusRatioY = (focusY - mHelperRect.top) / mHelperRect.height()
        mDrawableScale = scale
        getDisplayDrawableBounds(mHelperRect)
        val scaledFocusX = mHelperRect.left + focusRatioX * mHelperRect.width()
        val scaledFocusY = mHelperRect.top + focusRatioY * mHelperRect.height()
        mDisplayDrawableLeft += focusX - scaledFocusX
        mDisplayDrawableTop += focusY - scaledFocusY
        updateGrid()
        invalidate()
    }

    private val mSettleAnimatorUpdateListener =
        AnimatorUpdateListener { animation ->
            val animatedValue = animation.animatedValue as Float
            getDisplayDrawableBounds(mHelperRect)
            val overScrollX = measureOverScrollX(mHelperRect)
            val overScrollY = measureOverScrollY(mHelperRect)
            val overScale = measureOverScale()
            mDisplayDrawableLeft -= overScrollX * animatedValue
            mDisplayDrawableTop -= overScrollY * animatedValue
            val targetScale = mDrawableScale / overScale
            val newScale = (1 - animatedValue) * mDrawableScale + animatedValue * targetScale
            setScaleKeepingFocus(newScale, mScaleFocusX, mScaleFocusY)
            updateGrid()
            invalidate()
        }

    companion object {
        const val DEFAULT_MINIMUM_RATIO = 4f / 5f
        const val DEFAULT_MAXIMUM_RATIO = 1.91f
        const val DEFAULT_RATIO = 1f
        private const val MAXIMUM_OVER_SCROLL = 144f
        private const val MAXIMUM_OVER_SCALE = 0.7f
        private const val SET_BACK_DURATION: Long = 400
    }
}