package com.siguiente.fasal.imageCrop

import android.animation.ValueAnimator
import android.animation.ValueAnimator.AnimatorUpdateListener
import android.annotation.SuppressLint
import android.annotation.TargetApi
import android.content.Context
import android.graphics.Bitmap
import android.graphics.Canvas
import android.graphics.PointF
import android.graphics.RectF
import android.graphics.drawable.Drawable
import android.net.Uri
import android.os.Build
import android.util.AttributeSet
import android.util.Log
import android.view.GestureDetector
import android.view.MotionEvent
import android.view.ScaleGestureDetector
import android.view.ScaleGestureDetector.OnScaleGestureListener
import android.view.View
import android.view.animation.DecelerateInterpolator
import kotlin.math.*


class InstaCropperViewNew : View {
    interface BitmapCallback {
        fun onBitmapReady(bitmap: Bitmap?)
    }

    private var mMinimumRatio = DEFAULT_MINIMUM_RATIO
    private var mMaximumRatio = DEFAULT_MAXIMUM_RATIO
    private var mDefaultRatio = DEFAULT_RATIO
    private var mImageUri: Uri? = null
    private var mImageRawWidth = 0
    private var mImageRawHeight = 0
    private var mMakeDrawableTask: MakeDrawableTask? = null
    private var mWidth = 0
    private var mHeight = 0
    private val mGridDrawable = GridDrawable()
    private var mDrawable: Drawable? = null
    private var mFocusX = 0f
    private var mFocusY = 0f
    private var mRectangle: Rectangle? = null
    private val mViewBounds = RectF()
    private val mFitness = Fitness()
    private val mHelperFix = Fix()
    private val mHelperRect = RectF()
    private var mGestureDetector: GestureDetector? = null
    private var mScaleGestureDetector: ScaleGestureDetector? = null
    private var mMaximumOverScroll = 0f
    private var mAnimator: ValueAnimator? = null

    constructor(context: Context) : super(context) {
        initialize(context, null, 0, 0)
    }

    constructor(context: Context, attrs: AttributeSet?) : super(context, attrs) {
        initialize(context, attrs, 0, 0)
    }

    constructor(context: Context, attrs: AttributeSet?, defStyleAttr: Int) : super(
        context,
        attrs,
        defStyleAttr
    ) {
        initialize(context, attrs, defStyleAttr, 0)
    }

    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    constructor(
        context: Context,
        attrs: AttributeSet?,
        defStyleAttr: Int,
        defStyleRes: Int
    ) : super(context, attrs, defStyleAttr, defStyleRes) {
        initialize(context, attrs, defStyleAttr, defStyleRes)
    }

    private fun initialize(
        context: Context,
        attrs: AttributeSet?,
        defStyleAttr: Int,
        defStyleRes: Int
    ) {
        mGestureDetector = GestureDetector(context, mOnGestureListener)
        mScaleGestureDetector = ScaleGestureDetector(context, mOnScaleGestureListener)
        mMaximumOverScroll = resources.displayMetrics.density * MAXIMUM_OVER_SCROLL
        mAnimator = ValueAnimator()
        mAnimator!!.duration = SET_BACK_DURATION
        mAnimator!!.setFloatValues(0f, 1f)
        mAnimator!!.interpolator = DecelerateInterpolator(0.25f)
        mAnimator!!.addUpdateListener(mSettleAnimatorUpdateListener)
        mGridDrawable.callback = mGridCallback
    }

    private val mGridCallback: Drawable.Callback = object : Drawable.Callback {
        override fun invalidateDrawable(who: Drawable) {
            invalidate()
        }

        override fun scheduleDrawable(who: Drawable, what: Runnable, `when`: Long) {}
        override fun unscheduleDrawable(who: Drawable, what: Runnable) {}
    }

    fun setRatios(defaultRatio: Float, minimumRatio: Float, maximumRatio: Float) {
        mDefaultRatio = defaultRatio
        mMinimumRatio = minimumRatio
        mMaximumRatio = maximumRatio
        if (mAnimator!!.isRunning) {
            mAnimator!!.cancel()
        }
        cancelMakingDrawableProcessIfExists()
        mDrawable = null
        requestLayout()
    }

    fun setImageUri(uri: Uri?) {
        cancelMakingDrawableProcessIfExists()
        mImageUri = uri
        requestLayout()
        invalidate()
    }

    // TODO
    var drawableRotation: Float
        get() = mRectangle!!.rotation
        set(rotation) {
            if (rotation == mRectangle!!.rotation) {
                return
            }

            // TODO
            invalidate()
        }

    override fun onMeasure(widthMeasureSpec: Int, heightMeasureSpec: Int) {
        val widthMode = MeasureSpec.getMode(widthMeasureSpec)
        val widthSize = MeasureSpec.getSize(widthMeasureSpec)
        val heightMode = MeasureSpec.getMode(heightMeasureSpec)
        val heightSize = MeasureSpec.getSize(heightMeasureSpec)
        var targetWidth = 1
        var targetHeight = 1
        when (widthMode) {
            MeasureSpec.EXACTLY -> {
                targetWidth = widthSize
                when (heightMode) {
                    MeasureSpec.EXACTLY -> targetHeight = heightSize
                    MeasureSpec.AT_MOST -> targetHeight =
                        heightSize.coerceAtMost((targetWidth / mDefaultRatio).toInt())
                    MeasureSpec.UNSPECIFIED -> targetHeight = (targetWidth / mDefaultRatio).toInt()
                }
            }
            MeasureSpec.AT_MOST -> when (heightMode) {
                MeasureSpec.EXACTLY -> {
                    targetHeight = heightSize
                    targetWidth = widthSize.coerceAtMost((targetHeight * mDefaultRatio).toInt())
                }
                MeasureSpec.AT_MOST -> {
                    val specRatio = widthSize.toFloat() / heightSize.toFloat()
                    when {
                        specRatio == mDefaultRatio -> {
                            targetWidth = widthSize
                            targetHeight = heightSize
                        }
                        specRatio > mDefaultRatio -> {
                            targetHeight = heightSize
                            targetWidth = (targetHeight * mDefaultRatio).toInt()
                        }
                        else -> {
                            targetWidth = widthSize
                            targetHeight = (targetWidth / mDefaultRatio).toInt()
                        }
                    }
                }
                MeasureSpec.UNSPECIFIED -> {
                    targetWidth = widthSize
                    targetHeight = (targetWidth / mDefaultRatio).toInt()
                }
            }
            MeasureSpec.UNSPECIFIED -> when (heightMode) {
                MeasureSpec.EXACTLY -> {
                    targetHeight = heightSize
                    targetWidth = (targetHeight * mDefaultRatio).toInt()
                }
                MeasureSpec.AT_MOST -> {
                    targetHeight = heightSize
                    targetWidth = (targetHeight * mDefaultRatio).toInt()
                }
                MeasureSpec.UNSPECIFIED -> {
                    targetWidth = mMaximumOverScroll.toInt()
                    targetHeight = mMaximumOverScroll.toInt()
                }
            }
        }
        setMeasuredDimension(targetWidth, targetHeight)
    }

    override fun onLayout(changed: Boolean, left: Int, top: Int, right: Int, bottom: Int) {
        super.onLayout(changed, left, top, right, bottom)
        mWidth = right - left
        mHeight = bottom - top
        if (mWidth == 0 || mHeight == 0) {
            return
        }
        if (mImageUri == null) {
            return
        }
        mViewBounds[0f, 0f, mWidth.toFloat()] = mHeight.toFloat()
        if (currentDrawableIsSuitableForView()) {
            cancelMakingDrawableProcessIfExists()
            return
        }
        if (isMakingDrawableForView) {
            if (drawableBeingMadeIsSuitableForView()) {
                return
            }
            cancelMakingDrawableProcessIfExists()
        }
        startMakingSuitableDrawable()
    }

    private fun currentDrawableIsSuitableForView(): Boolean {
        if (mDrawable == null) {
            return false
        }
        val drawableWidth = mDrawable!!.intrinsicWidth
        val drawableHeight = mDrawable!!.intrinsicHeight
        return isSizeSuitableForView(drawableWidth, drawableHeight)
    }

    private fun cancelMakingDrawableProcessIfExists() {
        if (mMakeDrawableTask != null) {
            mMakeDrawableTask!!.cancel(true)
            mMakeDrawableTask = null
        }
    }

    private val isMakingDrawableForView: Boolean
        get() = mMakeDrawableTask != null

    private fun drawableBeingMadeIsSuitableForView(): Boolean {
        return isSizeSuitableForView(
            mMakeDrawableTask!!.targetWidth,
            mMakeDrawableTask!!.targetHeight
        )
    }

    private fun isSizeSuitableForView(width: Int, height: Int): Boolean {
        val viewArea = mWidth * mHeight
        val drawableArea = width * height
        val areaRatio = viewArea.toFloat() / drawableArea.toFloat()
        return areaRatio in 0.5f..2f
    }

    private fun startMakingSuitableDrawable() {
        mMakeDrawableTask = @SuppressLint("StaticFieldLeak")
        object : MakeDrawableTask(context, mImageUri!!, mWidth, mHeight) {
             override fun onPostExecute(result: Drawable?) {
                mDrawable = result
                mImageRawWidth = rawWidth
                mImageRawHeight = rawHeight
                onDrawableChanged()
            }
        }
        (mMakeDrawableTask as MakeDrawableTask).execute()
    }

    private fun onDrawableChanged() {
        reset()
    }

    private fun reset() {
        if (mAnimator!!.isRunning) {
            mAnimator!!.cancel()
        }
        mRectangle = Rectangle(
            mImageRawWidth.toFloat(),
            mImageRawHeight.toFloat(), mViewBounds.centerX(), mViewBounds.centerY()
        )
        mRectangle!!.getFitness(mViewBounds, mFitness)
        mFitness.getFittingFix(mHelperFix)
        mHelperFix.apply(mRectangle!!, mViewBounds, mMinimumRatio, mMaximumRatio)
        updateGrid()
        invalidate()
    }

    private fun updateGrid() {
        mHelperRect.set(mViewBounds)
        mHelperRect.intersect(0f, 0f, mWidth.toFloat(), mHeight.toFloat())
        setGridBounds(mHelperRect)
        invalidate()
    }

    private fun setGridBounds(bounds: RectF) {
        mGridDrawable.setBounds(
            bounds.left.toInt(), bounds.top.toInt(),
            bounds.right.toInt(), bounds.bottom.toInt()
        )
        invalidate()
    }

    override fun onDraw(canvas: Canvas) {
        super.onDraw(canvas)
        if (mDrawable == null) {
            return
        }
        canvas.save()
        canvas.translate(-mRectangle!!.centerX, -mRectangle!!.centerY)
        mGridDrawable.draw(canvas)
        canvas.scale(mRectangle!!.scale, mRectangle!!.scale, 0f, 0f)
        canvas.rotate(mRectangle!!.rotation, 0f, 0f)
        mDrawable!!.setBounds(0, 0, mRectangle!!.width.toInt(), mRectangle!!.height.toInt())
        mDrawable!!.draw(canvas)
        canvas.restore()
        Log.d(
            "AAA",
            "raw: " + mImageRawWidth + ", " + mImageRawHeight + " now: " + mRectangle!!.width + ", " + mRectangle!!.height
        )
    }

    override fun onTouchEvent(event: MotionEvent): Boolean {
        if (mDrawable == null) {
            return false
        }
        mGestureDetector!!.onTouchEvent(event)
        mScaleGestureDetector!!.onTouchEvent(event)
        val action = event.action
        if (action == MotionEvent.ACTION_UP || action == MotionEvent.ACTION_CANCEL || action == MotionEvent.ACTION_OUTSIDE) {
            mAnimator!!.start()
        }
        return true
    }

    private val mOnGestureListener: GestureDetector.OnGestureListener =
        object : GestureDetector.OnGestureListener {
            override fun onDown(motionEvent: MotionEvent): Boolean {
                return true
            }

            override fun onShowPress(motionEvent: MotionEvent) {}
            override fun onSingleTapUp(motionEvent: MotionEvent): Boolean {
                return false
            }

            override fun onScroll(
                e1: MotionEvent,
                e2: MotionEvent,
                distanceX: Float,
                distanceY: Float
            ): Boolean {
                var distanceX = distanceX
                var distanceY = distanceY
                distanceX = -distanceX
                distanceY = -distanceY
                mRectangle!!.getFitness(mViewBounds, mFitness)
                mFitness.getEssentialFix(mHelperFix)
                val overScrollX = mHelperFix.translateX
                val overScrollY = mHelperFix.translateY
                distanceX = applyOverScrollFix(distanceX, overScrollX)
                distanceY = applyOverScrollFix(distanceY, overScrollY)
                mRectangle!!.translateBy(distanceX, distanceY)
                updateGrid()
                invalidate()
                return true
            }

            override fun onLongPress(motionEvent: MotionEvent) {}
            override fun onFling(
                motionEvent: MotionEvent,
                motionEvent1: MotionEvent,
                v: Float,
                v1: Float
            ): Boolean {
                return false
            }
        }

    private fun applyOverScrollFix(distance: Float, overScroll: Float): Float {
        var distance = distance
        if (overScroll * distance <= 0) {
            return distance
        }
        val offRatio = abs(overScroll) / mMaximumOverScroll
        distance -= (distance * sqrt(offRatio.toDouble())).toFloat()
        return distance
    }

    private val mOnScaleGestureListener: OnScaleGestureListener = object : OnScaleGestureListener {
        override fun onScale(detector: ScaleGestureDetector): Boolean {
            val overScale = measureOverScale()
            val scale = applyOverScaleFix(detector.scaleFactor, overScale)
            mFocusX = detector.focusX
            mFocusY = detector.focusY
            mRectangle!!.scaleBy(scale, mFocusX, mFocusY)
            return true
        }

        override fun onScaleBegin(detector: ScaleGestureDetector): Boolean {
            return true
        }

        override fun onScaleEnd(detector: ScaleGestureDetector) {}
    }

    private fun measureOverScale(): Float {
        mRectangle!!.getFitness(mViewBounds, mFitness)
        mFitness.getEssentialFix(mHelperFix)
        return measureOverScale(mHelperFix)
    }

    private fun measureOverScale(fix: Fix): Float {
        val maximumAllowedScale = maximumAllowedScale
        val minimumAllowedScale = minimumAllowedScale
        val scale = fix.getScale(mRectangle)
        return when {
            scale < minimumAllowedScale -> {
                scale / minimumAllowedScale
            }
            scale > maximumAllowedScale -> {
                scale / maximumAllowedScale
            }
            else -> {
                1f
            }
        }
    }

    // TODO
    private val maximumAllowedScale: Float
        get() {
            val maximumAllowedWidth = mImageRawWidth.toFloat()
            val maximumAllowedHeight = mImageRawHeight.toFloat()
            // TODO
            return (maximumAllowedWidth / mWidth.toFloat()).coerceAtMost(maximumAllowedHeight / mHeight.toFloat())
        }

    private val minimumAllowedScale: Float
        get() {
            mRectangle!!.getFitness(mViewBounds, mFitness)
            mFitness.getFittingFix(mHelperFix)
            return mHelperFix.getScale(mRectangle)
        }

    private fun applyOverScaleFix(scale: Float, overScale: Float): Float {
        var scale = scale
        var overScale = overScale
        if (overScale == 1f) {
            return scale
        }
        if (overScale > 1) {
            overScale = 1f / overScale
        }
        var wentOverScaleRatio = (overScale - MAXIMUM_OVER_SCALE) / (1 - MAXIMUM_OVER_SCALE)
        if (wentOverScaleRatio < 0) {
            wentOverScaleRatio = 0f
        }

        // 1 -> scale , 0 -> 1
        // scale * f(1) = scale
        // scale * f(0) = 1

        // f(1) = 1
        // f(0) = 1/scale
        scale *= wentOverScaleRatio + (1 - wentOverScaleRatio) / scale
        return scale
    }

    private val mSettleAnimatorUpdateListener =
        AnimatorUpdateListener { animation ->
            val animatedValue = animation.animatedValue as Float
            mRectangle!!.getFitness(mViewBounds, mFitness)
            mFitness.getEssentialFix(mHelperFix)
            val overScrollX = mHelperFix.translateX
            val overScrollY = mHelperFix.translateY
            val overScale = measureOverScale(mHelperFix)
            val translateX = -overScrollX * animatedValue
            val translateY = -overScrollY * animatedValue
            mRectangle!!.translateBy(translateX, translateY)
            val scale = mRectangle!!.scale
            val targetScale = scale / overScale
            Log.d("AAA", "scale=$scale targetScale=$targetScale")
            val newScale = (1 - animatedValue) * scale + animatedValue * targetScale
            mRectangle!!.scaleBy(newScale / scale, mFocusX, mFocusY)
            updateGrid()
            invalidate()
        }

    private class Fix {
        var translateX = 0f
        var translateY = 0f
        var sizeChangeX = 0f
        var sizeChangeY = 0f
        fun apply(rectangle: Rectangle, inside: RectF?, minimumRatio: Float, maximumRatio: Float) {
            rectangle.translateBy(translateX, translateY)
            val rotation: Float = rectangle.rotation
            if (rotation == 0f || rotation == 180f) {
                val width = rectangle.width
                val height = rectangle.height
                val newWidth = width + sizeChangeX
                val newHeight = height + sizeChangeY

                // TODO
            }
            val scale = getScale(rectangle)
            rectangle.scaleBy(scale, rectangle.centerX, rectangle.centerY)
        }

        fun getScale(rectangle: Rectangle?): Float {
            val width = rectangle!!.width
            val height = rectangle.height
            val ratio = width / height
            val rotation: Float = rectangle.rotation
            val r = rotation / (2 * Math.PI)
            val sin = sin(-r)
            val cos = cos(-r)
            val widthChange = sizeChangeX * cos - sizeChangeY * sin
            val heightChange = sizeChangeX * sin + sizeChangeY * cos
            var newWidth = (width + widthChange).toFloat()
            var newHeight = (height + heightChange).toFloat()
            val newRatio = newWidth / newHeight
            if (newRatio < ratio) {
                newHeight = newWidth / ratio
            } else {
                newWidth = newHeight * ratio
            }
            return newWidth / width
        }

        override fun toString(): String {
            return "dx=$translateX dy=$translateY dSizeX=$sizeChangeX dSizeY=$sizeChangeY"
        }
    }

    private class Fitness {
        private val mEssentialBias = RectF()
        private val mOptionalBias = RectF()
        operator fun set(
            essentialInPositiveX: Float,
            essentialInNegativeX: Float,
            essentialInPositiveY: Float,
            essentialInNegativeY: Float,
            optionalInPositiveX: Float,
            optionalInNegativeX: Float,
            optionalInPositiveY: Float,
            optionalInNegativeY: Float
        ) {
            mEssentialBias[essentialInNegativeX, essentialInNegativeY, essentialInPositiveX] =
                essentialInPositiveY
            mOptionalBias[optionalInNegativeX, optionalInNegativeY, optionalInPositiveX] =
                optionalInPositiveY
            Log.d("AAA", "fitness set. " + toString())
        }

        fun getFittingFix(fix: Fix) {
            fix.translateX = mEssentialBias.centerX() + mOptionalBias.centerX()
            fix.translateY = mEssentialBias.centerY() + mOptionalBias.centerY()
            fix.sizeChangeX = mEssentialBias.width() - mOptionalBias.width()
            fix.sizeChangeY = mEssentialBias.height() - mOptionalBias.height()
            Log.d("AAA", "Fitting fix is: $fix")
        }

        fun getEssentialFix(fix: Fix) {
            if (mOptionalBias.left >= mEssentialBias.left && mOptionalBias.right <= mEssentialBias.right) {
                fix.translateX = mEssentialBias.centerX()
                fix.sizeChangeX = mEssentialBias.width()
            } else if (mOptionalBias.left <= mEssentialBias.left && mOptionalBias.right >= mEssentialBias.right) {
                fix.translateX = 0f
                fix.sizeChangeX = 0f
            } else if (mEssentialBias.left < mOptionalBias.left) {
                fix.translateX = mEssentialBias.left
                fix.sizeChangeX = 0f.coerceAtLeast(mEssentialBias.right - mOptionalBias.right)
            } else if (mEssentialBias.right > mOptionalBias.right) {
                fix.translateX = mEssentialBias.right
                fix.sizeChangeX = 0f.coerceAtLeast(mOptionalBias.left - mEssentialBias.left)
            }
            if (mOptionalBias.top >= mEssentialBias.top && mOptionalBias.bottom <= mEssentialBias.bottom) {
                fix.translateY = mEssentialBias.centerY()
                fix.sizeChangeY = mEssentialBias.height()
            } else if (mOptionalBias.top <= mEssentialBias.top && mOptionalBias.bottom >= mEssentialBias.bottom) {
                fix.translateY = 0f
                fix.sizeChangeY = 0f
            } else if (mEssentialBias.top < mOptionalBias.top) {
                fix.translateY = mEssentialBias.top
                fix.sizeChangeY = 0f.coerceAtLeast(mEssentialBias.bottom - mOptionalBias.bottom)
            } else if (mEssentialBias.bottom > mOptionalBias.bottom) {
                fix.translateY = mEssentialBias.bottom
                fix.sizeChangeY = 0f.coerceAtLeast(mOptionalBias.top - mEssentialBias.top)
            }
        }

        override fun toString(): String {
            return "Essential bias: $mEssentialBias Optional bias: $mOptionalBias"
        }
    }

    private class Rectangle(
        var width: Float,
        var height: Float,
        var centerX: Float,
        var centerY: Float
    ) {
        var scale = 1f
            private set
        var rotation = 0f
        private val mLines: Array<Line?> = arrayOfNulls(4)
        fun getFitness(bounds: RectF, fitness: Fitness) {
            var essentialInPositiveX = 0f
            var essentialInNegativeX = 0f
            var essentialInPositiveY = 0f
            var essentialInNegativeY = 0f
            var optionalInPositiveX = 0f
            var optionalInNegativeX = 0f
            var optionalInPositiveY = 0f
            var optionalInNegativeY = 0f
            for (line in mLines) {
                val lineFitness = line!!.getFitness(bounds)
                //                Log.d("AAA", "Line fitness: " + lineFitness);
                val isEssential = lineFitness < 0
                val dx = lineFitness * line.directionX
                val dy = lineFitness * line.directionY
                if (isEssential) {
                    if (dx > 0) {
                        essentialInPositiveX = essentialInPositiveX.coerceAtLeast(dx)
                    } else {
                        essentialInNegativeX = essentialInNegativeX.coerceAtMost(dx)
                    }
                    if (dy > 0) {
                        essentialInPositiveY = essentialInPositiveY.coerceAtLeast(dy)
                    } else {
                        essentialInNegativeY = essentialInNegativeY.coerceAtMost(dy)
                    }
                } else {
                    if (dx > 0) {
                        optionalInPositiveX = optionalInPositiveX.coerceAtMost(dx)
                    } else {
                        optionalInNegativeX = optionalInNegativeX.coerceAtLeast(dx)
                    }
                    if (dy > 0) {
                        optionalInPositiveY = optionalInPositiveY.coerceAtMost(dy)
                    } else {
                        optionalInNegativeY = optionalInNegativeY.coerceAtLeast(dy)
                    }
                }
            }
            fitness[essentialInPositiveX, essentialInNegativeX, essentialInPositiveY, essentialInNegativeY, optionalInPositiveX, optionalInNegativeX, optionalInPositiveY] =
                optionalInNegativeY
        }

        private fun updateRotation() {
            while (rotation >= 360) {
                rotation -= 360f
            }
            while (rotation < 0) {
                rotation += 360f
            }
        }


        fun translateBy(dx: Float, dy: Float) {
            centerX += dx
            centerY += dy
            for (line in mLines) {
                line!!.translateBy(dx, dy)
            }
        }

        fun scaleBy(scale: Float, pivotX: Float, pivotY: Float) {
            this.scale *= scale
            width *= scale
            height *= scale
            for (line in mLines) {
                val fitness = line!!.getFitness(pivotX, pivotY)
                val translateX = (scale - 1) * fitness * line.directionX
                val translateY = (scale - 1) * fitness * line.directionY
                line.translateBy(translateX, translateY)
            }
            calcCenter()
        }

        private fun calcCenter() {
            var sumX = 0f
            var sumY = 0f
            for (line in mLines) {
                sumX += line!!.x
                sumY += line.y
            }
            centerX = sumX / mLines.size
            centerY = sumY / mLines.size
        }


        init {
            mLines[0] = Line(centerX - width / 2, centerY - height / 2, 1f, 0f)
            mLines[1] = Line(centerX - width / 2, centerY - height / 2, 0f, 1f)
            mLines[2] = Line(centerX + width / 2, centerY + height / 2, (-1).toFloat(), 0f)
            mLines[3] = Line(centerX + width / 2, centerY + height / 2, 0f, (-1).toFloat())
        }
    }

    private open class Line(var x: Float, var y: Float, var directionX: Float, var directionY: Float) {
        fun getFitness(bounds: RectF): Float {
            val lt = getFitness(bounds.left, bounds.top)
            val rt = getFitness(bounds.right, bounds.top)
            val rb = getFitness(bounds.right, bounds.bottom)
            val lb = getFitness(bounds.left, bounds.bottom)
            return lt.coerceAtMost(rt).coerceAtMost(rb.coerceAtMost(lb))
        }

        fun getFitness(pointX: Float, pointY: Float): Float {
            // x = x - dy*t , y = y + dx*t
            // x = pointX + dx*q , y = pointY + dy*q
            val q = directionX * (x - pointX) + directionY * (y - pointY)
            val crossX = pointX + directionX * q
            val crossY = pointY + directionY * q
            val distance = PointF.length(crossX - pointX, crossY - pointY)
            return -sign(q) * distance
        }

        protected fun rotateBy(sin: Double, cos: Double) {
            val newDirectionX = directionX * cos - directionY * sin
            val newDirectionY = directionX * sin + directionY * cos
            directionX = newDirectionX.toFloat()
            directionY = newDirectionY.toFloat()
        }

        fun translateBy(dx: Float, dy: Float) {
            x += dx
            y += dy
        }
    }

    companion object {
        const val DEFAULT_MINIMUM_RATIO = 4f / 5f
        const val DEFAULT_MAXIMUM_RATIO = 1.91f
        const val DEFAULT_RATIO = 1f
        private const val MAXIMUM_OVER_SCROLL = 144f
        private const val MAXIMUM_OVER_SCALE = 0.7f
        private const val SET_BACK_DURATION: Long = 400
    }
}