package com.siguiente.fasal.imageCrop

import android.annotation.SuppressLint
import android.content.Context
import android.graphics.*
import android.graphics.drawable.BitmapDrawable
import android.graphics.drawable.Drawable
import android.media.ExifInterface
import android.net.Uri
import android.os.AsyncTask
import android.os.Build
import android.provider.MediaStore
import android.util.Log
import androidx.annotation.RequiresApi
import java.io.FileNotFoundException
import java.io.IOException


/**
 * Created by Yashar on 3/8/2017.
 */
open class MakeDrawableTask protected constructor(
    val mContext: Context,
    private val mUri: Uri,
    val targetWidth: Int,
     val targetHeight: Int
) :
    AsyncTask<Void?, Void?, Drawable?>() {
    var rawWidth = 0
    var rawHeight = 0

    @RequiresApi(Build.VERSION_CODES.Q)
    protected override fun doInBackground(vararg params: Void?): Drawable? {
        val options = BitmapFactory.Options()
        options.inSampleSize = 1
        options.inJustDecodeBounds = true
        return try {
            BitmapFactory.decodeStream(
                mContext.contentResolver.openInputStream(mUri),
                null,
                options
            )
            rawWidth = options.outWidth
            rawHeight = options.outHeight
            var resultWidth = rawWidth
            var resultHeight = rawHeight
            Runtime.getRuntime().gc()
            val totalMemory = Runtime.getRuntime().maxMemory()
            val allowedMemoryToUse = totalMemory / 8
            val maximumAreaPossibleAccordingToAvailableMemory = (allowedMemoryToUse / 4).toInt()
            val targetArea = (targetWidth * targetHeight * 4).coerceAtMost(
                maximumAreaPossibleAccordingToAvailableMemory
            )
            var resultArea = resultWidth * resultHeight
            while (resultArea > targetArea) {
                options.inSampleSize *= 2
                resultWidth = rawWidth / options.inSampleSize
                resultHeight = rawHeight / options.inSampleSize
                resultArea = resultWidth * resultHeight
            }
            options.inJustDecodeBounds = false
            val bitmap = getBitmap(mContext, mUri, options) ?: return null
            val beforeRatio = rawWidth.toFloat() / rawHeight.toFloat()
            val afterRatio = bitmap.width.toFloat() / bitmap.height.toFloat()
            if (beforeRatio < 1 && afterRatio > 1 || beforeRatio > 1 && afterRatio < 1) {
                val rawWidth = rawWidth
                this.rawWidth = rawHeight
                rawHeight = rawWidth
            }
            BitmapDrawable(mContext.resources, bitmap)
        } catch (e: FileNotFoundException) {
            null
        }
    }

    companion object {
        private const val TAG = "MakeDrawableTask"
        @RequiresApi(Build.VERSION_CODES.Q)
        fun getBitmap(
            context: Context,
            uri: Uri,
            options: BitmapFactory.Options
        ): Bitmap? {
            var bitmap: Bitmap? = null
            while (true) {
                try {
                    bitmap = BitmapFactory.decodeStream(
                        context.contentResolver.openInputStream(uri),
                        null,
                        options
                    )
                    break
                } catch (t: Throwable) {
                    options.inSampleSize *= 2
                    if (options.inSampleSize >= 1024) {
                        Log.d(TAG, "Failed to optimize RAM to receive Bitmap.")
                        break
                    }
                }
            }
            if (bitmap != null) {
                val orientation = getRotation(uri, context)
                if (orientation != 0) {
                    val matrix = Matrix()
                    matrix.postRotate(orientation.toFloat())
                    bitmap = Bitmap.createBitmap(
                        bitmap,
                        0,
                        0,
                        bitmap.width,
                        bitmap.height,
                        matrix,
                        false
                    )
                }
            }
            return bitmap
        }

        @RequiresApi(Build.VERSION_CODES.Q)
        private fun getRotation(uri: Uri, context: Context): Int {
            if (isUriMatching(MediaStore.Images.Media.EXTERNAL_CONTENT_URI, uri) || isUriMatching(
                    MediaStore.Images.Media.INTERNAL_CONTENT_URI,
                    uri
                )
            ) {
                val c = context.contentResolver.query(
                    uri,
                    arrayOf(MediaStore.Images.Media.ORIENTATION),
                    null,
                    null,
                    null
                )
                return if (c!!.count == 1) {
                    c.moveToFirst()
                    val orientation = c.getInt(0)
                    c.close()
                    orientation
                } else {
                    Log.w(TAG, "Failed to get MediaStore image orientation.")
                    c.close()
                    0
                }
            }
            return try {
                val ei: ExifInterface = if (Build.VERSION.SDK_INT >= 24) {
                    ExifInterface(context.contentResolver.openInputStream(uri)!!)
                } else {
                    ExifInterface(uri.toString())
                }
                val orientation = ei.getAttributeInt(
                    ExifInterface.TAG_ORIENTATION,
                    ExifInterface.ORIENTATION_UNDEFINED
                )
                when (orientation) {
                    ExifInterface.ORIENTATION_ROTATE_90 -> 90
                    ExifInterface.ORIENTATION_ROTATE_180 -> 180
                    ExifInterface.ORIENTATION_ROTATE_270 -> 270
                    ExifInterface.ORIENTATION_NORMAL -> 0
                    else -> 0
                }
            } catch (e: IOException) {
                Log.w(TAG, "Failed to get image orientation from file.", e)
                0
            }
        }

        fun resizeBitmap(bitmap: Bitmap, newWidth: Int, newHeight: Int): Bitmap {
            val resizedBitmap = Bitmap.createBitmap(newWidth, newHeight, Bitmap.Config.ARGB_8888)
            val scaleX = newWidth / bitmap.width.toFloat()
            val scaleY = newHeight / bitmap.height.toFloat()
            val pivotX = 0f
            val pivotY = 0f
            val scaleMatrix = Matrix()
            scaleMatrix.setScale(scaleX, scaleY, pivotX, pivotY)
            val canvas = Canvas(resizedBitmap)
            canvas.setMatrix(scaleMatrix)
            canvas.drawBitmap(bitmap, 0f, 0f, Paint(Paint.FILTER_BITMAP_FLAG))
            return resizedBitmap
        }

        private fun isUriMatching(path: Uri, element: Uri): Boolean {
            return Uri.withAppendedPath(path, element.lastPathSegment) == element
        }
    }
}