package com.siguiente.fasal.utils

import android.app.Activity
import android.content.Context
import android.content.Intent
import android.graphics.*
import android.net.Uri
import android.os.Build
import android.view.View
import android.view.inputmethod.InputMethodManager
import android.widget.*
import androidx.annotation.RequiresApi
import androidx.core.content.ContextCompat
import androidx.databinding.BindingAdapter
import androidx.fragment.app.FragmentManager
import com.bumptech.glide.Glide
import com.bumptech.glide.load.engine.DiskCacheStrategy
import com.google.android.exoplayer2.*
import com.google.android.exoplayer2.source.ExtractorMediaSource
import com.google.android.exoplayer2.trackselection.DefaultTrackSelector
import com.google.android.exoplayer2.ui.PlayerView
import com.google.android.exoplayer2.upstream.DefaultHttpDataSourceFactory
import com.google.android.material.snackbar.Snackbar
import com.google.android.material.textfield.TextInputLayout
import com.siguiente.fasal.home.model.CategoryModel
import com.siguiente.fasal.interfaces.PlayerStateCallback
import com.siguiente.fasal.my_store.model.SubCategoryModel
import com.siguiente.fasal.profile.ProfileFragment
import com.siguiente.fasal.utils.SpinnerExtensions.ItemSelectedListener
import com.siguiente.fasal.utils.SpinnerExtensions.setSpinnerEntries
import com.siguiente.fasal.utils.SpinnerExtensions.setSpinnerItemSelectedListener
import com.siguiente.fasal.utils.SpinnerExtensions.setSubCategory
import com.siguiente.fasal.utils.SpinnerExtensions.setSubSpinnerItemSelectedListener
import java.text.SimpleDateFormat
import java.util.*
import kotlin.collections.ArrayList


fun Context.toast(message: String) {
    Toast.makeText(this, message, Toast.LENGTH_LONG).show()
}

@BindingAdapter("entries")
fun Spinner.setEntries(entries: ArrayList<CategoryModel>?) {
    setSpinnerEntries(entries)
}

@BindingAdapter("subEntries")
fun Spinner.setSubCategoryData(subentries: ArrayList<SubCategoryModel>?) {
    setSubCategory(subentries)
}


@BindingAdapter("onItemSelected")
fun Spinner.setItemSelectedListener(itemSelectedListener: ItemSelectedListener?) {
    setSpinnerItemSelectedListener(itemSelectedListener)
}


@BindingAdapter("onItemSelected")
fun Spinner.setSubItemSelectedListener(itemSelectedListener: ItemSelectedListener?) {
    setSubSpinnerItemSelectedListener(itemSelectedListener)
}

@BindingAdapter("app:errorText")
fun setErrorMessage(view: TextInputLayout, errorMessage: String) {
    view.error = errorMessage
}

@BindingAdapter("thumb")
fun loadImage(view: ImageView, url: String) {
    Glide.with(view)
        .load(url)
        .diskCacheStrategy(DiskCacheStrategy.NONE)
        .skipMemoryCache(true)
        .into(view)
}

@BindingAdapter("video_url", "on_state_change")
fun PlayerView.loadVideo(url: String, callback: PlayerStateCallback) {
    val player = ExoPlayerFactory.newSimpleInstance(
        context, DefaultRenderersFactory(context), DefaultTrackSelector(),
        DefaultLoadControl()
    )

    player.playWhenReady = true
    player.repeatMode = Player.REPEAT_MODE_ALL
    // When changing track, retain the latest frame instead of showing a black screen
    setKeepContentOnPlayerReset(true)
    // We'll show the controller
    this.useController = true
    // Provide url to load the video from here
    val mediaSource = ExtractorMediaSource.Factory(
        DefaultHttpDataSourceFactory("Demo")
    ).createMediaSource(Uri.parse(url))

    player.prepare(mediaSource)

    this.player = player

    this.player!!.addListener(object : Player.EventListener {

        override fun onPlayerError(error: ExoPlaybackException) {
            super.onPlayerError(error)
            this@loadVideo.context.toast("Oops! Error occurred while playing media.")
        }

        override fun onPlayerStateChanged(playWhenReady: Boolean, playbackState: Int) {
            super.onPlayerStateChanged(playWhenReady, playbackState)

            if (playbackState == Player.STATE_BUFFERING) callback.onVideoBuffering(player) // Buffering.. set progress bar visible here
            if (playbackState == Player.STATE_READY) {
                // [PlayerView] has fetched the video duration so this is the block to hide the buffering progress bar
                callback.onVideoDurationRetrieved(
                    (this@loadVideo.player as SimpleExoPlayer).duration,
                    player
                )
            }
            if (playbackState == Player.STATE_READY && player.playWhenReady) {
                // [PlayerView] has started playing/resumed the video
                callback.onStartedPlaying(player)
            }
        }
    })
}

@BindingAdapter("bindFormattedDate")
fun bindFormattedDate(textView: TextView, dateInput: String?) {
    val fromServer = SimpleDateFormat("yyyy-MM-dd")
    val myFormat = SimpleDateFormat("EEE, MMM d, yy")
    val date: Date = fromServer.parse(dateInput)
    val outputDateStr: String = myFormat.format(date)
    textView.text = outputDateStr
}


fun Context.openProfile(fragmentSupportManager: FragmentManager) {
    val profileFragment = ProfileFragment()
    profileFragment.show(fragmentSupportManager, "Profile")
}

fun Context.fetchColor(id: Int): Int = ContextCompat.getColor(this, id)

fun <A : Activity> Activity.startNewActivity(activity: Class<A>) {
    Intent(this, activity).also {
        it.flags = Intent.FLAG_ACTIVITY_NEW_TASK or Intent.FLAG_ACTIVITY_CLEAR_TASK
        startActivity(it)
    }
}

fun View.visible(isVisible: Boolean) {
    visibility = if (isVisible) View.VISIBLE else View.GONE
}

fun View.enable(enabled: Boolean) {
    isEnabled = enabled
    alpha = if (enabled) 1f else 0.5f
}

fun Context.hideKeyboard(view: View) {
    val inputMethodManager = getSystemService(Activity.INPUT_METHOD_SERVICE) as InputMethodManager
    inputMethodManager.hideSoftInputFromWindow(view.windowToken, 0)
}


fun View.snackbar(message: String, action: (() -> Unit)? = null) {
    val snackbar = Snackbar.make(this, message, Snackbar.LENGTH_LONG)
    action?.let {
        snackbar.setAction("Retry") {
            it()
        }
    }
    snackbar.show()
}

fun getCircularBitmap(srcBitmap: Bitmap): Bitmap? {
    // Calculate the circular bitmap width with border
    val squareBitmapWidth = Math.min(srcBitmap.width, srcBitmap.height)
    // Initialize a new instance of Bitmap
    val dstBitmap = Bitmap.createBitmap(
        squareBitmapWidth,  // Width
        squareBitmapWidth,  // Height
        Bitmap.Config.ARGB_8888 // Config
    )
    // Initialize a new Canvas to draw circular bitmap
    val canvas = Canvas(dstBitmap)
    // Initialize a new Paint instance
    val paint = Paint()
    paint.isAntiAlias = true
    // Initialize a new Rect instance
    val rect = Rect(0, 0, squareBitmapWidth, squareBitmapWidth)
    // Initialize a new RectF instance
    val rectF = RectF(rect)
    // Draw an oval shape on Canvas
    canvas.drawOval(rectF, paint)
    paint.xfermode = PorterDuffXfermode(PorterDuff.Mode.SRC_IN)
    // Calculate the left and top of copied bitmap
    val left = ((squareBitmapWidth - srcBitmap.width) / 2).toFloat()
    val top = ((squareBitmapWidth - srcBitmap.height) / 2).toFloat()
    // Make a rounded image by copying at the exact center position of source image
    canvas.drawBitmap(srcBitmap, left, top, paint)
    // Free the native object associated with this bitmap.
    srcBitmap.recycle()
    // Return the circular bitmap
    return dstBitmap
}

@RequiresApi(Build.VERSION_CODES.M)
fun createImage(width: Int, height: Int, colorBG: Int, colorText: Int, name: String?): Bitmap? {
    val bitmap = Bitmap.createBitmap(width, height, Bitmap.Config.ARGB_8888)
    val canvas = Canvas(bitmap)
    val paint2 = Paint()
    paint2.setColor(colorBG)
    canvas.drawRect(0f, 0f, width.toFloat(), height.toFloat(), paint2)
    val paint = Paint()
    paint.setColor(colorText)
    paint.setTextSize(50f)
    paint.setTextScaleX(1f)
    canvas.drawText(name!!, 50f - 15, 50f + 15, paint)
    return getCircularBitmap(bitmap)
}


fun ProgressBar.show() {
    visibility = View.VISIBLE
}

fun ProgressBar.hide() {
    visibility = View.GONE
}

fun View.snackbar(message: String) {
    Snackbar.make(this, message, Snackbar.LENGTH_LONG).also { snackbar ->
        snackbar.setAction("Ok") {
            snackbar.dismiss()
        }
    }.show()
}
