<?php

namespace Faker\Provider\fa_IR;

class Text extends \Faker\Provider\Text
{
    /**
     * generates text string in arabic
     *
     * @example 'از تاریخ‌الشعرا را بکوبند روی نبش دیوار کوچه‌شان. تابلوی مدرسه.'
     * @param  int                   $maxNbChars
     * @param  int                   $indexSize
     * @return string
     * @throws \InvalidArgumentException
     */
    public function realText($maxNbChars = 200, $indexSize = 2)
    {
        if ($maxNbChars < 10) {
            throw new \InvalidArgumentException('maxNbChars must be at least 10');
        }

        if ($indexSize < 1) {
            throw new \InvalidArgumentException('indexSize must be at least 1');
        }

        if ($indexSize > 5) {
            throw new \InvalidArgumentException('indexSize must be at most 5');
        }

        $words = $this->getConsecutiveWords($indexSize);
        $result = [];
        $resultLength = 0;
        // take a random starting point
        $next = static::randomKey($words);
        while ($resultLength < $maxNbChars && isset($words[$next])) {
            // fetch a random word to append
            $word = static::randomElement($words[$next]);

            // calculate next index
            $currentWords = explode(' ', $next);

            $currentWords[] = $word;
            array_shift($currentWords);
            $next = implode(' ', $currentWords);

            if ($resultLength == 0 && !preg_match('/^[\x{0600}-\x{06FF}]/u', $word)) {
                continue;
            }
            // append the element
            $result[] = $word;
            $resultLength += strlen($word) + 1;
        }

        // remove the element that caused the text to overflow
        array_pop($result);

        // build result
        $result = implode(' ', $result);

        return $result . '.';
    }

    /**
     * License: Creative Commons Attribution-ShareAlike License
     *
     * Title: مدیر مدرسه
     * Author: جلال آل‌احمد
     * Language: Persian
     *
     * @see http://fa.wikisource.org/wiki/%D9%85%D8%AF%DB%8C%D8%B1_%D9%85%D8%AF%D8%B1%D8%B3%D9%87
     * @var string
     */
    protected static $baseText = <<<'EOT'
از در که وارد شدم سیگارم دستم بود. زورم آمد سلام کنم. همین طوری دنگم گرفته بود قد باشم. رئیس فرهنگ که اجازه‌ی نشستن داد، نگاهش لحظه‌ای روی دستم مکث کرد و بعد چیزی را که می‌نوشت، تمام کرد و می‌خواست متوجه من بشود که رونویس حکم را روی میزش گذاشته بودم. حرفی نزدیم. رونویس را با کاغذهای ضمیمه‌اش زیر و رو کرد و بعد غبغب انداخت و آرام و مثلاً خالی از عصبانیت گفت:

- جا نداریم آقا. این که نمی‌شه! هر روز یه حکم می‌دند دست یکی می‌فرستنش سراغ من... دیروز به آقای مدیر کل...

حوصله‌ی این اباطیل را نداشتم. حرفش را بریدم که:

- ممکنه خواهش کنم زیر همین ورقه مرقوم بفرمایید؟

و سیگارم را توی زیرسیگاری براق روی میزش تکاندم. روی میز، پاک و مرتب بود. درست مثل اتاق همان مهمان‌خانه‌ی تازه‌عروس‌ها. هر چیز به جای خود و نه یک ذره گرد. فقط خاکستر سیگار من زیادی بود. مثل تفی در صورت تازه تراشیده‌ای.... قلم را برداشت و زیر حکم چیزی نوشت و امضا کرد و من از در آمده بودم بیرون. خلاص. تحمل این یکی را نداشتم. با اداهایش. پیدا بود که تازه رئیس شده. زورکی غبغب می‌انداخت و حرفش را آهسته توی چشم آدم می‌زد. انگار برای شنیدنش گوش لازم نیست. صد و پنجاه تومان در کارگزینی کل مایه گذاشته بودم تا این حکم را به امضا رسانده بودم. توصیه هم برده بودم و تازه دو ماه هم دویده بودم. مو، لای درزش نمی‌رفت. می‌دانستم که چه او بپذیرد، چه نپذیرد، کار تمام است. خودش هم می‌دانست. حتماً هم دستگیرش شد که با این نک و نالی که می‌کرد، خودش را کنف کرده. ولی کاری بود و شده بود. در کارگزینی کل، سفارش کرده بودند که برای خالی نبودن عریضه رونویس را به رؤیت رئیس فرهنگ هم برسانم تازه این طور شد. و گر نه بالی حکم کارگزینی کل چه کسی می‌توانست حرفی بزند؟ یک وزارت خانه بود و یک کارگزینی! شوخی که نبود. ته دلم قرص‌تر از این‌ها بود که محتاج به این استدلال‌ها باشم. اما به نظرم همه‌ی این تقصیرها از این سیگار لعنتی بود که به خیال خودم خواسته بودم خرجش را از محل اضافه حقوق شغل جدیدم در بیاورم. البته از معلمی، هم اُقم نشسته بود. ده سال «الف.ب.» درس دادن و قیافه‌های بهت‌زده‌ی بچه‌های مردم برای مزخرف‌ترین چرندی که می‌گویی... و استغناء با غین و استقراء با قاف و خراسانی و هندی و قدیمی‌ترین شعر دری و صنعت ارسال مثل و ردالعجز... و از این مزخرفات! دیدم دارم خر می‌شوم. گفتم مدیر بشوم. مدیر دبستان! دیگر نه درس خواهم داد و نه مجبور خواهم بود برای فرار از اتلاف وقت، در امتحان تجدیدی به هر احمق بی‌شعوری هفت بدهم تا ایام آخر تابستانم را که لذیذترین تکه‌ی تعطیلات است، نجات داده باشم. این بود که راه افتادم. رفتم و از اهلش پرسیدم. از یک کار چاق کن. دستم را توی دست کارگزینی گذاشت و قول و قرار و طرفین خوش و خرم و یک روز هم نشانی مدرسه را دستم دادند که بروم وارسی، که باب میلم هست یا نه.

و رفتم. مدرسه دو طبقه بود و نوساز بود و در دامنه‌ی کوه تنها افتاده بود و آفتاب‌رو بود. یک فرهنگ‌دوست خرپول، عمارتش را وسط زمین خودش ساخته بود و بیست و پنج سال هم در اختیار فرهنگ گذاشته بود که مدرسه‌اش کنند و رفت و آمد بشود و جاده‌ها کوبیده بشود و این قدر ازین بشودها بشود، تا دل ننه باباها بسوزد و برای این‌که راه بچه‌هاشان را کوتاه بکنند، بیایند همان اطراف مدرسه را بخرند و خانه بسازند و زمین یارو از متری یک عباسی بشود صد تومان. یارو اسمش را هم روی دیوار مدرسه کاشی‌کاری کرده بود. هنوز در و همسایه پیدا نکرده بودند که حرف‌شان بشود و لنگ و پاچه‌ی سعدی و باباطاهر را بکشند میان و یک ورق دیگر از تاریخ‌الشعرا را بکوبند روی نبش دیوار کوچه‌شان. تابلوی مدرسه هم حسابی و بزرگ و خوانا. از صد متری داد می‌زد که توانا بود هر.... هر چه دلتان بخواهد! با شیر و خورشیدش که آن بالا سر، سه پا ایستاده بود و زورکی تعادل خودش را حفظ می‌کرد و خورشید خانم روی کولش با ابروهای پیوسته و قمچیلی که به دست داشت و تا سه تیر پرتاب، اطراف مدرسه بیابان بود. درندشت و بی آب و آبادانی و آن ته رو به شمال، ردیف کاج‌های درهم فرو رفته‌ای که از سر دیوار گلی یک باغ پیدا بود روی آسمان لکه‌ی دراز و تیره‌ای زده بود. حتماً تا بیست و پنج سال دیگر همه‌ی این اطراف پر می‌شد و بوق ماشین و ونگ ونگ بچه‌ها و فریاد لبویی و زنگ روزنامه‌فروشی و عربده‌ی گل به سر دارم خیار! نان یارو توی روغن بود.

- راستی شاید متری ده دوازده شاهی بیشتر نخریده باشد؟ شاید هم زمین‌ها را همین جوری به ثبت داده باشد؟ هان؟

- احمق به توچه؟!...

بله این فکرها را همان روزی کردم که ناشناس به مدرسه سر زدم و آخر سر هم به این نتیجه رسیدم که مردم حق دارند جایی بخوابند که آب زیرشان نرود.

- تو اگر مردی، عرضه داشته باش مدیر همین مدرسه هم بشو.

و رفته بودم و دنبال کار را گرفته بودم تا رسیده بودم به این‌جا. همان روز وارسی فهمیده بودم که مدیر قبلی مدرسه زندانی است. لابد کله‌اش بوی قرمه‌سبزی می‌داده و باز لابد حالا دارد کفاره‌ی گناهانی را می‌دهد که یا خودش نکرده یا آهنگری در بلخ کرده. جزو پر قیچی‌های رئیس فرهنگ هم کسی نبود که با مدیرشان، اضافه حقوقی نصیبش بشود و ناچار سر و دستی برای این کار بشکند. خارج از مرکز هم نداشت. این معلومات را توی کارگزینی به دست آورده بودم. هنوز «گه خوردم نامه‌نویسی» هم مد نشده بود که بگویم یارو به این زودی‌ها از سولدونی در خواهد آمد. فکر نمی‌کردم که دیگری هم برای این وسط بیابان دلش لک زده باشد با زمستان سختش و با رفت و آمد دشوارش.

این بود که خیالم راحت بود. از همه‌ی این‌ها گذشته کارگزینی کل موافقت کرده بود! دست است که پیش از بلند شدن بوی اسکناس، آن جا هم دو سه تا عیب شرعی و عرفی گرفته بودند و مثلاً گفته بودن لابد کاسه‌ای زیر نیم کاسه است که فلانی یعنی من، با ده سال سابقه‌ی تدریس، می‌خواهد مدیر دبستان بشود! غرض‌شان این بود که لابد خل شدم که از شغل مهم و محترم دبیری دست می‌شویم. ماهی صد و پنجاه تومان حق مقام در آن روزها پولی نبود که بتوانم نادیده بگیرم. و تازه اگر ندیده می‌گرفتم چه؟ باز باید بر می‌گشتم به این کلاس‌ها و این جور حماقت‌ها. این بود که پیش رئیس فرهنگ، صاف برگشتم به کارگزینی کل، سراغ آن که بفهمی نفهمی، دلال کارم بود. و رونویس حکم را گذاشتم و گفتم که چه طور شد و آمدم بیرون.

دو روز بعد رفتم سراغش. معلوم شد که حدسم درست بوده است و رئیس فرهنگ گفته بوده: «من از این لیسانسه‌های پر افاده نمی‌خواهم که سیگار به دست توی هر اتاقی سر می‌کنند.»

و یارو برایش گفته بود که اصلاً وابدا..! فلانی همچین و همچون است و مثقالی هفت صنار با دیگران فرق دارد و این هندوانه‌ها و خیال من راحت باشد و پنج‌شنبه یک هفته‌ی دیگر خودم بروم پهلوی او... و این کار را کردم. این بار رئیس فرهنگ جلوی پایم بلند شد که: «ای آقا... چرا اول نفرمودید؟!...» و از کارمندهایش گله کرد و به قول خودش، مرا «در جریان موقعیت محل» گذاشت و بعد با ماشین خودش مرا به مدرسه رساند و گفت زنگ را زودتر از موعد زدند و در حضور معلم‌ها و ناظم، نطق غرایی در خصائل مدیر جدید – که من باشم – کرد و بعد هم مرا گذاشت و رفت با یک مدرسه‌ی شش کلاسه‌ی «نوبنیاد» و یک ناظم و هفت تا معلم و دویست و سی و پنج تا شاگرد. دیگر حسابی مدیر مدرسه شده بودم!

ناظم، جوان رشیدی بود که بلند حرف می‌زد و به راحتی امر و نهی می‌کرد و بیا و برویی داشت و با شاگردهای درشت، روی هم ریخته بود که خودشان ترتیب کارها را می‌دادند و پیدا بود که به سر خر احتیاجی ندارد و بی‌مدیر هم می‌تواند گلیم مدرسه را از آب بکشد. معلم کلاس چهار خیلی گنده بود. دو تای یک آدم حسابی. توی دفتر، اولین چیزی که به چشم می‌آمد. از آن‌هایی که اگر توی کوچه ببینی، خیال می‌کنی مدیر کل است. لفظ قلم حرف می‌زد و شاید به همین دلیل بود که وقتی رئیس فرهنگ رفت و تشریفات را با خودش برد، از طرف همکارانش تبریک ورود گفت و اشاره کرد به اینکه «ان‌شاءالله زیر سایه‌ی سرکار، سال دیگر کلاس‌های دبیرستان را هم خواهیم داشت.» پیدا بود که این هیکل کم‌کم دارد از سر دبستان زیادی می‌کند! وقتی حرف می‌زد همه‌اش درین فکر بودم که با نان آقا معلمی چه طور می‌شد چنین هیکلی به هم زد و چنین سر و تیپی داشت؟ و راستش تصمیم گرفتم که از فردا صبح به صبح ریشم را بتراشم و یخه‌ام تمیز باشد و اتوی شلوارم تیز.

معلم کلاس اول باریکه‌ای بود، سیاه سوخته. با ته ریشی و سر ماشین کرده‌ای و یخه‌ی بسته. بی‌کراوات. شبیه میرزابنویس‌های دم پست‌خانه. حتی نوکر باب می‌نمود. و من آن روز نتوانستم بفهمم وقتی حرف می‌زند کجا را نگاه می‌کند. با هر جیغ کوتاهی که می‌زد هرهر می‌خندید. با این قضیه نمی‌شد کاری کرد. معلم کلاس سه، یک جوان ترکه‌ای بود؛ بلند و با صورت استخوانی و ریش از ته تراشیده و یخه‌ی بلند آهاردار. مثل فرفره می‌جنبید. چشم‌هایش برق عجیبی می‌زد که فقط از هوش نبود، چیزی از ناسلامتی در برق چشم‌هایش بود که مرا واداشت از ناظم بپرسم مبادا مسلول باشد. البته مسلول نبود، تنها بود و در دانشگاه درس می‌خواند. کلاس‌های پنجم و ششم را دو نفر با هم اداره می‌کردند. یکی فارسی و شرعیات و تاریخ، جغرافی و کاردستی و این جور سرگرمی‌ها را می‌گفت، که جوانکی بود بریانتین زده، با شلوار پاچه تنگ و پوشت و کراوات زرد و پهنی که نعش یک لنگر بزرگ آن را روی سینه‌اش نگه داشته بود و دائماً دستش حمایل موهای سرش بود و دم به دم توی شیشه‌ها نگاه می‌کرد. و آن دیگری که حساب و مرابحه و چیزهای دیگر می‌گفت، جوانی بود موقر و سنگین مازندرانی به نظر می‌آمد و به خودش اطمینان داشت. غیر از این‌ها، یک معلم ورزش هم داشتیم که دو هفته بعد دیدمش و اصفهانی بود و از آن قاچاق‌ها.

رئیس فرهنگ که رفت، گرم و نرم از همه‌شان حال و احوال پرسیدم. بعد به همه سیگار تعارف کردم. سراپا همکاری و همدردی بود. از کار و بار هر کدامشان پرسیدم. فقط همان معلم کلاس سه، دانشگاه می‌رفت. آن که لنگر به سینه انداخته بود، شب‌ها انگلیسی می‌خواند که برود آمریکا. چای و بساطی در کار نبود و ربع ساعت‌های تفریح، فقط توی دفتر جمع می‌شدند و دوباره از نو. و این نمی‌شد. باید همه‌ی سنن را رعایت کرد. دست کردم و یک پنج تومانی روی میز گذاشتم و قرار شد قبل و منقلی تهیه کنند و خودشان چای را راه بیندازند.

بعد از زنگ قرار شد من سر صف نطقی بکنم. ناظم قضیه را در دو سه کلمه برای بچه‌ها گفت که من رسیدم و همه دست زدند. چیزی نداشتم برایشان بگویم. فقط یادم است اشاره‌ای به این کردم که مدیر خیلی دلش می‌خواست یکی از شما را به جای فرزند داشته باشد و حالا نمی‌داند با این همه فرزند چه بکند؟! که بی‌صدا خندیدند و در میان صف‌های عقب یکی پکی زد به خنده. واهمه برم داشت که «نه بابا. کار ساده‌ای هم نیست!» قبلاً فکر کرده بودم که می‌روم و فارغ از دردسر اداره‌ی کلاس، در اتاق را روی خودم می‌بندم و کار خودم را می‌کنم. اما حالا می‌دیدم به این سادگی‌ها هم نیست. اگر فردا یکی‌شان زد سر اون یکی را شکست، اگر یکی زیر ماشین رفت؛ اگر یکی از ایوان افتاد؛ چه خاکی به سرم خواهم ریخت؟

حالا من مانده بودم و ناظم که چیزی از لای در آهسته خزید تو. کسی بود؛ فراش مدرسه با قیافه‌ای دهاتی و ریش نتراشیده و قدی کوتاه و گشاد گشاد راه می‌رفت و دست‌هایش را دور از بدن نگه می‌داشت. آمد و همان کنار در ایستاد. صاف توی چشمم نگاه می‌کرد. حال او را هم پرسیدم. هر چه بود او هم می‌توانست یک گوشه‌ی این بار را بگیرد. در یک دقیقه همه‌ی درد دل‌هایش را کرد و التماس دعاهایش که تمام شد، فرستادمش برایم چای درست کند و بیاورد. بعد از آن من به ناظم پرداختم. سال پیش، از دانشسرای مقدماتی در آمده بود. یک سال گرمسار و کرج کار کرده بود و امسال آمده بود این‌جا. پدرش دو تا زن داشته. از اولی دو تا پسر که هر دو تا چاقوکش از آب در آمده‌اند و از دومی فقط او مانده بود که درس‌خوان شده و سرشناس و نان مادرش را می‌دهد که مریض است و از پدر سال‌هاست که خبری نیست و... یک اتاق گرفته‌اند به پنجاه تومان و صد و پنجاه تومان حقوق به جایی نمی‌رسد و تازه زور که بزند سه سال دیگر می‌تواند از حق فنی نظامت مدرسه استفاده کند

... بعد بلند شدیم که به کلاس‌ها سرکشی کنیم. بعد با ناظم به تک تک کلاس‌ها سر زدیم در این میان من به یاد دوران دبستان خودم افتادم. در کلاس ششم را باز کردیم «... ت بی پدرو مادر» جوانک بریانتین زده خورد توی صورت‌مان. یکی از بچه‌ها صورتش مثل چغندر قرمز بود. لابد بزک فحش هنوز باقی بود. قرائت فارسی داشتند. معلم دستهایش توی جیبش بود و سینه‌اش را پیش داده بود و زبان به شکایت باز کرد:

- آقای مدیر! اصلاً دوستی سرشون نمی‌شه. تو سَری می‌خوان. ملاحظه کنید بنده با چه صمیمیتی...

حرفش را در تشدید «ایت» بریدم که:

- صحیح می‌فرمایید. این بار به من ببخشید.

و از در آمدیم بیرون. بعد از آن به اطاقی که در آینده مال من بود سر زدیم. بهتر از این نمی‌شد. بی سر و صدا، آفتاب‌رو، دور افتاده.

وسط حیاط، یک حوض بزرگ بود و کم‌عمق. تنها قسمت ساختمان بود که رعایت حال بچه‌های قد و نیم قد در آن شده بود. دور حیاط دیوار بلندی بود درست مثل دیوار چین. سد مرتفعی در مقابل فرار احتمالی فرهنگ و ته حیاط مستراح و اتاق فراش بغلش و انبار زغال و بعد هم یک کلاس. به مستراح هم سر کشیدیم. همه بی در و سقف و تیغه‌ای میان آن‌ها. نگاهی به ناظم کردم که پا به پایم می‌آمد. گفت:

- دردسر عجیبی شده آقا. تا حالا صد تا کاغذ به ادارفردا صبح رفتم مدرسه. بچه‌ها با صف‌هاشان به طرف کلاس‌ها می‌رفتند و ناظم چوب به دست توی ایوان ایستاده بود و توی دفتر دو تا از معلم‌ها بودند. معلوم شد کار هر روزه‌شان است. ناظم را هم فرستادم سر یک کلاس دیگر و خودم آمدم دم در مدرسه به قدم زدن؛ فکر کردم از هر طرف که بیایند مرا این ته، دم در مدرسه خواهند دید و تمام طول راه در این خجالت خواهند ماند و دیگر دیر نخواهند آمد. یک سیاهی از ته جاده‌ی جنوبی پیداشد. جوانک بریانتین زده بود. مسلماً او هم مرا می‌دید، ولی آهسته‌تر از آن می‌آمد که یک معلم تأخیر کرده جلوی مدیرش می‌آمد. جلوتر که آمد حتی شنیدم که سوت می‌زد. اما بی‌انصاف چنان سلانه سلانه می‌آمد که دیدم هیچ جای گذشت نیست. اصلاً محل سگ به من نمی‌گذاشت. داشتم از کوره در می‌رفتم که یک مرتبه احساس کردم تغییری در رفتار خود داد و تند کرد.

به خیر گذشت و گرنه خدا عالم است چه اتفاقی می‌افتاد. سلام که کرد مثل این که می‌خواست چیزی بگوید که پیش دستی کردم:

- بفرمایید آقا. بفرمایید، بچه‌ها منتظرند.

واقعاً به خیر گذشت. شاید اتوبوسش دیر کرده. شاید راه‌بندان بوده؛ جاده قرق بوده و باز یک گردن‌کلفتی از اقصای عالم می‌آمده که ازین سفره‌ی مرتضی علی بی‌نصیب نماند. به هر صورت در دل بخشیدمش. چه خوب شد که بد و بی‌راهی نگفتی! که از دور علم افراشته‌ی هیکل معلم کلاس چهارم نمایان شد. از همان ته مرا دیده بود. تقریباً می‌دوید. تحمل این یکی را نداشتم. «بدکاری می‌کنی. اول بسم‌الله و مته به خشخاش!» رفتم و توی دفتر نشستم و خودم را به کاری مشغول کردم که هن هن کنان رسید. چنان عرق از پیشانی‌اش می‌ریخت که راستی خجالت کشیدم. یک لیوان آب از کوه به دستش دادم و مسخ‌شده‌ی خنده‌اش را با آب به خوردش دادم و بلند که شد برود، گفتم:

- عوضش دو کیلو لاغر شدید.

برگشت نگاهی کرد و خنده‌ای و رفت. ناگهان ناظم از در وارد شد و از را ه نرسیده گفت:

- دیدید آقا! این جوری می‌آند مدرسه. اون قرتی که عین خیالش هم نبود آقا! اما این یکی...

از او پرسیدم:

- انگار هنوز دو تا از کلاس‌ها ولند؟

- بله آقا. کلاس سه ورزش دارند. گفتم بنشینند دیکته بنویسند آقا. معلم حساب پنج و شش هم که نیومده آقا.

در همین حین یکی از عکس‌های بزرگ دخمه‌های هخامنشی را که به دیوار کوبیده بود پس زد و:

- نگاه کنید آقا...

روی گچ دیوار با مداد قرمز و نه چندان درشت، به عجله و ناشیانه علامت داس کشیده بودند. همچنین دنبال کرد:

- از آثار دوره‌ی اوناست آقا. کارشون همین چیزها بود. روزنومه بفروشند. تبلیغات کنند و داس چکش بکشند آقا. رئیس‌شون رو که گرفتند چه جونی کندم آقا تا حالی‌شون کنم که دست ور دارند آقا. و از روی میز پرید پایین.

- گفتم مگه باز هم هستند؟

- آره آقا، پس چی! یکی همین آقازاده که هنوز نیومده آقا. هر روز نیم ساعت تأخیر داره آقا. یکی هم مثل کلاس سه.

- خوب چرا تا حالا پاکش نکردی؟

- به! آخه آدم درد دلشو واسه‌ی کی بگه؟ آخه آقا در میان تو روی آدم می‌گند جاسوس، مأمور! باهاش حرفم شده آقا. کتک و کتک‌کاری!

و بعد یک سخنرانی که چه طور مدرسه را خراب کرده‌اند و اعتماد اهل محله را چه طور از بین برده‌اند که نه انجمنی، نه کمکی به بی‌بضاعت‌ها؛ و از این حرف ها.

بعد از سخنرانی آقای ناظم دستمالم را دادم که آن عکس‌ها را پاک کند و بعد هم راه افتادم که بروم سراغ اتاق خودم. در اتاقم را که باز کردم، داشتم دماغم با بوی خاک نم کشیده‌اش اخت می‌کرد که آخرین معلم هم آمد. آمدم توی ایوان و با صدای بلند، جوری که در تمام مدرسه بشنوند، ناظم را صدا زدم و گفتم با قلم قرمز برای آقا یک ساعت تأخیر بگذارند.ه‌ی ساختمان نوشتیم آقا. می‌گند نمی‌شه پول دولت رو تو ملک دیگرون خرج کرد.

- گفتم راست می‌گند.

دیگه کافی بود. آمدیم بیرون. همان توی حیاط تا نفسی تازه کنیم وضع مالی و بودجه و ازین حرف‌های مدرسه را پرسیدم. هر اتاق ماهی پانزده ریال حق نظافت داشت. لوازم‌التحریر و دفترها را هم اداره‌ی فرهنگ می‌داد. ماهی بیست و پنج تومان هم برای آب خوردن داشتند که هنوز وصول نشده بود. برای نصب هر بخاری سالی سه تومان. ماهی سی تومان هم تنخواه‌گردان مدرسه بود که مثل پول آب سوخت شده بود و حالا هم ماه دوم سال بود. اواخر آبان. حالیش کردم که حوصله‌ی این کارها را ندارم و غرضم را از مدیر شدن برایش خلاصه کردم و گفتم حاضرم همه‌ی اختیارات را به او بدهم. «اصلاً انگار که هنوز مدیر نیامده.» مهر مدرسه هم پهلوی خودش باشد. البته او را هنوز نمی‌شناختم. شنیده بودم که مدیرها قبلاً ناظم خودشان را انتخاب می‌کنند، اما من نه کسی را سراغ داشتم و نه حوصله‌اش را. حکم خودم را هم به زور گرفته بودم. سنگ‌هامان را وا کندیم و به دفتر رفتیم و چایی را که فراش از بساط خانه‌اش درست کرده بود، خوردیم تا زنگ را زدند و باز هم زدند و من نگاهی به پرونده‌های شاگردها کردم که هر کدام عبارت بود از دو برگ کاغذ. از همین دو سه برگ کاغذ دانستم که اولیای بچه‌ها اغلب زارع و باغبان و اویارند و قبل از این‌که زنگ آخر را بزنند و مدرسه تعطیل بشود بیرون آمدم. برای روز اول خیلی زیاد بود.

فردا صبح رفتم مدرسه. بچه‌ها با صف‌هاشان به طرف کلاس‌ها می‌رفتند و ناظم چوب به دست توی ایوان ایستاده بود و توی دفتر دو تا از معلم‌ها بودند. معلوم شد کار هر روزه‌شان است. ناظم را هم فرستادم سر یک کلاس دیگر و خودم آمدم دم در مدرسه به قدم زدن؛ فکر کردم از هر طرف که بیایند مرا این ته، دم در مدرسه خواهند دید و تمام طول راه در این خجالت خواهند ماند و دیگر دیر نخواهند آمد. یک سیاهی از ته جاده‌ی جنوبی پیداشد. جوانک بریانتین زده بود. مسلماً او هم مرا می‌دید، ولی آهسته‌تر از آن می‌آمد که یک معلم تأخیر کرده جلوی مدیرش می‌آمد. جلوتر که آمد حتی شنیدم که سوت می‌زد. اما بی‌انصاف چنان سلانه سلانه می‌آمد که دیدم هیچ جای گذشت نیست. اصلاً محل سگ به من نمی‌گذاشت. داشتم از کوره در می‌رفتم که یک مرتبه احساس کردم تغییری در رفتار خود داد و تند کرد.

به خیر گذشت و گرنه خدا عالم است چه اتفاقی می‌افتاد. سلام که کرد مثل این که می‌خواست چیزی بگوید که پیش دستی کردم:

- بفرمایید آقا. بفرمایید، بچه‌ها منتظرند.

واقعاً به خیر گذشت. شاید اتوبوسش دیر کرده. شاید راه‌بندان بوده؛ جاده قرق بوده و باز یک گردن‌کلفتی از اقصای عالم می‌آمده که ازین سفره‌ی مرتضی علی بی‌نصیب نماند. به هر صورت در دل بخشیدمش. چه خوب شد که بد و بی‌راهی نگفتی! که از دور علم افراشته‌ی هیکل معلم کلاس چهارم نمایان شد. از همان ته مرا دیده بود. تقریباً می‌دوید. تحمل این یکی را نداشتم. «بدکاری می‌کنی. اول بسم‌الله و مته به خشخاش!» رفتم و توی دفتر نشستم و خودم را به کاری مشغول کردم که هن هن کنان رسید. چنان عرق از پیشانی‌اش می‌ریخت که راستی خجالت کشیدم. یک لیوان آب از کوه به دستش دادم و مسخ‌شده‌ی خنده‌اش را با آب به خوردش دادم و بلند که شد برود، گفتم:

- عوضش دو کیلو لاغر شدید.

برگشت نگاهی کرد و خنده‌ای و رفت. ناگهان ناظم از در وارد شد و از را ه نرسیده گفت:

- دیدید آقا! این جوری می‌آند مدرسه. اون قرتی که عین خیالش هم نبود آقا! اما این یکی...

از او پرسیدم:

- انگار هنوز دو تا از کلاس‌ها ولند؟

- بله آقا. کلاس سه ورزش دارند. گفتم بنشینند دیکته بنویسند آقا. معلم حساب پنج و شش هم که نیومده آقا.

در همین حین یکی از عکس‌های بزرگ دخمه‌های هخامنشی را که به دیوار کوبیده بود پس زد و:

- نگاه کنید آقا...

روی گچ دیوار با مداد قرمز و نه چندان درشت، به عجله و ناشیانه علامت داس کشیده بودند. همچنین دنبال کرد:

- از آثار دوره‌ی اوناست آقا. کارشون همین چیزها بود. روزنومه بفروشند. تبلیغات کنند و داس چکش بکشند آقا. رئیس‌شون رو که گرفتند چه جونی کندم آقا تا حالی‌شون کنم که دست ور دارند آقا. و از روی میز پرید پایین.

- گفتم مگه باز هم هستند؟

- آره آقا، پس چی! یکی همین آقازاده که هنوز نیومده آقا. هر روز نیم ساعت تأخیر داره آقا. یکی هم مثل کلاس سه.

- خوب چرا تا حالا پاکش نکردی؟

- به! آخه آدم درد دلشو واسه‌ی کی بگه؟ آخه آقا در میان تو روی آدم می‌گند جاسوس، مأمور! باهاش حرفم شده آقا. کتک و کتک‌کاری!

و بعد یک سخنرانی که چه طور مدرسه را خراب کرده‌اند و اعتماد اهل محله را چه طور از بین برده‌اند که نه انجمنی، نه کمکی به بی‌بضاعت‌ها؛ و از این حرف ها.

بعد از سخنرانی آقای ناظم دستمالم را دادم که آن عکس‌ها را پاک کند و بعد هم راه افتادم که بروم سراغ اتاق خودم. در اتاقم را که باز کردم، داشتم دماغم با بوی خاک نم کشیده‌اش اخت می‌کرد که آخرین معلم هم آمد. آمدم توی ایوان و با صدای بلند، جوری که در تمام مدرسه بشنوند، ناظم را صدا زدم و گفتم با قلم قرمز برای آقا یک ساعت تأخیر بگذارند.

روز سوم باز اول وقت مدرسه بودم. هنوز از پشت دیوار نپیچیده بودم که صدای سوز و بریز بچه‌ها به پیشبازم آمد. تند کردم. پنج تا از بچه‌ها توی ایوان به خودشان می‌پیچیدند و ناظم ترکه‌ای به دست داشت و به نوبت به کف دست‌شان می‌زد. بچه‌ها التماس می‌کردند؛ گریه می‌کردند؛ اما دستشان را هم دراز می‌کردند. نزدیک بود داد بزنم یا با لگد بزنم و ناظم را پرت کنم آن طرف. پشتش به من بود و من را نمی‌دید. ناگهان زمزمه‌ای توی صف‌ها افتاد که یک مرتبه مرا به صرافت انداخت که در مقام مدیریت مدرسه، به سختی می‌شود ناظم را کتک زد. این بود که خشمم را فرو خوردم و آرام از پله‌ها رفتم بالا. ناظم، تازه متوجه من شده بود در همین حین دخالتم را کردم و خواهش کردم این بار همه‌شان را به من ببخشند.

نمی‌دانم چه کار خطایی از آنها سر زده بود که ناظم را تا این حد عصبانی کرده بود. بچه‌ها سکسکه‌کنان رفتند توی صف‌ها و بعد زنگ را زدند و صف‌ها رفتند به کلاس‌ها و دنبالشان هم معلم‌ها که همه سر وقت حاضر بودند. نگاهی به ناظم انداختم که تازه حالش سر جا آمده بود و گفتم در آن حالی که داشت، ممکن بود گردن یک کدامشان را بشکند. که مرتبه براق شد:

- اگه یک روز جلوشونو نگیرید سوارتون می‌شند آقا. نمی‌دونید چه قاطرهای چموشی شده‌اند آقا.

مثل بچه مدرسه‌ای‌ها آقا آقا می‌کرد. موضوع را برگرداندم و احوال مادرش را پرسیدم. خنده، صورتش را از هم باز کرد و صدا زد فراش برایش آب بیاورد. یادم هست آن روز نیم ساعتی برای آقای ناظم صحبت کردم. پیرانه. و او جوان بود و زود می‌شد رامش کرد. بعد ازش خواستم که ترکه‌ها را بشکند و آن وقت من رفتم سراغ اتاق خودم.

در همان هفته‌ی اول به کارها وارد شدم. فردای زمستان و نه تا بخاری زغال سنگی و روزی چهار بار آب آوردن و آب و جاروی اتاق‌ها با یک فراش جور در نمی‌آید. یک فراش دیگر از اداره ی فرهنگ خواستم که هر روز منتظر ورودش بودیم. بعد از ظهرها را نمی‌رفتم. روزهای اول با دست و دل لرزان، ولی سه چهار روزه جرأت پیدا کردم. احساس می‌کردم که مدرسه زیاد هم محض خاطر من نمی‌گردد. کلاس اول هم یکسره بود و به خاطر بچه‌های جغله دلهره‌ای نداشتم. در بیابان‌های اطراف مدرسه هم ماشینی آمد و رفت نداشت و گرچه پست و بلند بود اما به هر صورت از حیاط مدرسه که بزرگ‌تر بود. معلم ها هم، هر بعد از ظهری دو تاشان به نوبت می‌رفتند یک جوری باهم کنار آمده بودند. و ترسی هم از این نبود که بچه‌ها از علم و فرهنگ ثقل سرد بکنند. یک روز هم بازرس آمد و نیم ساعتی پیزر لای پالان هم گذاشتیم و چای و احترامات متقابل! و در دفتر بازرسی تصدیق کرد که مدرسه «با وجود عدم وسایل» بسیار خوب اداره می‌شود.

بچه‌ها مدام در مدرسه زمین می‌خوردند، بازی می‌کردند، زمین می‌خوردند. مثل اینکه تاتوله خورده بودند. ساده‌ترین شکل بازی‌هایشان در ربع ساعت‌های تفریح، دعوا بود. فکر می‌کردم علت این همه زمین خوردن شاید این باشد که بیش‌ترشان کفش حسابی ندارند. آن‌ها هم که داشتند، بچه‌ننه بودند و بلد نبودند بدوند و حتی راه بروند. این بود که روزی دو سه بار، دست و پایی خراش بر می‌داشت. پرونده‌ی برق و تلفن مدرسه را از بایگانی بسیار محقر مدرسه بیرون کشیده بودم و خوانده بودم. اگر یک خرده می‌دویدی تا دو سه سال دیگر هم برق مدرسه درست می‌شد و هم تلفنش. دوباره سری به اداره ساختمان زدم و موضوع را تازه کردم و به رفقایی که دورادور در اداره‌ی برق و تلفن داشتم، یکی دو بار رو انداختم که اول خیال می‌کردند کار خودم را می‌خواهم به اسم مدرسه راه بیندازم و ناچار رها کردم. این قدر بود که ادای وظیفه‌ای می‌کرد. مدرسه آب نداشت. نه آب خوراکی و نه آب جاری. با هرزاب بهاره، آب انبار زیر حوض را می‌انباشتند که تلمبه‌ای سرش بود و حوض را با همان پر می‌کردند و خود بچه‌ها. اما برای آب خوردن دو تا منبع صد لیتری داشتیم از آهن سفید که مثل امامزاده‌ای یا سقاخانه‌ای دو قلو، روی چهار پایه کنار حیاط بود و روزی دو بار پر و خالی می‌شد. این آب را از همان باغی می‌آوردیم که ردیف کاج‌هایش روی آسمان، لکه‌ی دراز سیاه انداخته بود. البته فراش می‌آورد. با یک سطل بزرگ و یک آب‌پاش که سوراخ بود و تا به مدرسه می‌رسید، نصف شده بود. هر دو را از جیب خودم دادم تعمیر کردند.

یک روز هم مالک مدرسه آمد. پیرمردی موقر و سنگین که خیال می‌کرد برای سرکشی به خانه‌ی مستأجرنشینش آمده. از در وارد نشده فریادش بلند شد و فحش را کشید به فراش و به فرهنگ که چرا بچه‌ها دیوار مدرسه را با زغال سیاه کرده‌اند واز همین توپ و تشرش شناختمش. کلی با او صحبت کردیم البته او دو برابر سن من را داشت. برایش چای هم آوردیم و با معلم‌ها آشنا شد و قول‌ها داد و رفت. کنه‌ای بود. درست یک پیرمرد. یک ساعت و نیم درست نشست. ماهی یک بار هم این برنامه را داشتند که بایست پیه‌اش را به تن می‌مالیدم.

اما معلم‌ها. هر کدام یک ابلاغ بیست و چهار ساعته در دست داشتند، ولی در برنامه به هر کدام‌شان بیست ساعت درس بیشتر نرسیده بود. کم کم قرار شد که یک معلم از فرهنگ بخواهیم و به هر کدام‌شان هجده ساعت درس بدهیم، به شرط آن‌که هیچ بعد از ظهری مدرسه تعطیل نباشد. حتی آن که دانشگاه می‌رفت می‌توانست با هفته‌ای هجده ساعت درس بسازد. و دشوارترین کار همین بود که با کدخدامنشی حل شد و من یک معلم دیگر از فرهنگ خواستم.

اواخر هفته‌ی دوم، فراش جدید آمد. مرد پنجاه ساله‌ای باریک و زبر و زرنگ که شب‌کلاه می‌گذاشت و لباس آبی می‌پوشید و تسبیح می‌گرداند و از هر کاری سر رشته داشت. آب خوردن را نوبتی می‌آوردند. مدرسه تر و تمیز شد و رونقی گرفت. فراش جدید سرش توی حساب بود. هر دو مستخدم با هم تمام بخاری‌ها را راه انداختند و یک کارگر هم برای کمک به آن‌ها آمد. فراش قدیمی را چهار روز پشت سر هم، سر ظهر می‌فرستادیم اداره‌ی فرهنگ و هر آن منتظر زغال بودیم. هنوز یک هفته از آمدن فراش جدید نگذشته بود که صدای همه‌ی معلم‌ها در آمده بود. نه به هیچ کدامشان سلام می‌کرد و نه به دنبال خرده فرمایش‌هایشان می‌رفت. درست است که به من سلام می‌کرد، اما معلم‌ها هم، لابد هر کدام در حدود من صاحب فضایل و عنوان و معلومات بودند که از یک فراش مدرسه توقع سلام داشته باشند. اما انگار نه انگار.

بدتر از همه این که سر خر معلم‌ها بود. من که از همان اول، خرجم را سوا کرده بودم و آن‌ها را آزاد گذاشته بودم که در مواقع بیکاری در دفتر را روی خودشان ببندند و هر چه می‌خواهند بگویند و هر کاری می‌خواهند بکنند. اما او در فاصله‌ی ساعات درس، همچه که معلم‌ها می‌آمدند، می‌آمد توی دفتر و همین طوری گوشه‌ی اتاق می‌ایستاد و معلم‌ها کلافه می‌شدند. نه می‌توانستند شلکلک‌های معلمی‌شان را در حضور او کنار بگذارند و نه جرأت می‌کردند به او چیزی بگویند. بدزبان بود و از عهده‌ی همه‌شان بر می‌آمد. یکی دوبار دنبال نخود سیاه فرستاده بودندش. اما زرنگ بود و فوری کار را انجام می‌داد و بر می‌گشت. حسابی موی دماغ شده بود. ده سال تجربه این حداقل را به من آموخته بود که اگر معلم‌ها در ربع ساعت‌های تفریح نتوانند بخندند، سر کلاس، بچه‌های مردم را کتک خواهند زد. این بود که دخالت کردم. یک روز فراش جدید را صدا زدم. اول حال و احوالپرسی و بعد چند سال سابقه دارد و چند تا بچه و چه قدر می‌گیرد... که قضیه حل شد. سی صد و خرده‌ای حقوق می‌گرفت. با بیست و پنج سال سابقه. کار از همین جا خراب بود. پیدا بود که معلم‌ها حق دارند او را غریبه بدانند. نه دیپلمی، نه کاغذپاره‌ای، هر چه باشد یک فراش که بیشتر نبود! و تازه قلدر هم بود و حق هم داشت. اول به اشاره و کنایه و بعد با صراحت بهش فهماندم که گر چه معلم جماعت اجر دنیایی ندارد، اما از او که آدم متدین و فهمیده‌ای است بعید است و از این حرف‌ها... که یک مرتبه پرید توی حرفم که:

- ای آقا! چه می‌فرمایید؟ شما نه خودتون این کاره‌اید و نه اینارو می‌شناسید. امروز می‌خواند سیگار براشون بخرم، فردا می‌فرستنم سراغ عرق. من این‌ها رو می‌شناسم.

راست می‌گفت. زودتر از همه، او دندان‌های مرا شمرده بود. فهمیده بود که در مدرسه هیچ‌کاره‌ام. می‌خواستم کوتاه بیایم، ولی مدیر مدرسه بودن و در مقابل یک فراش پررو ساکت ماندن!... که خر خر کامیون زغال به دادم رسید. ترمز که کرد و صدا خوابید گفتم:

- این حرف‌ها قباحت داره. معلم جماعت کجا پولش به عرق می‌رسه؟ حالا بدو زغال آورده‌اند.

و همین طور که داشت بیرون می‌رفت، افزودم:

- دو روز دیگه که محتاجت شدند و ازت قرض خواستند با هم رفیق می‌شید.

و آمدم توی ایوان. در بزرگ آهنی مدرسه را باز کرده بودند و کامیون آمده بود تو و داشتند بارش را جلوی انبار ته حیاط خالی می‌کردند و راننده، کاغذی به دست ناظم داد که نگاهی به آن انداخت و مرا نشان داد که در ایوان بالا ایستاده بودم و فرستادش بالا. کاغذش را با سلام به دستم داد. بیجک زغال بود. رسید رسمی اداره‌ی فرهنگ بود در سه نسخه و روی آن ورقه‌ی ماشین شده‌ی «باسکول» که می‌گفت کامیون و محتویاتش جمعاً دوازده خروار است. اما رسیدهای رسمی اداری فرهنگ ساکت بودند. جای مقدار زغالی که تحویل مدرسه داده شده بود، در هر سه نسخه خالی بود. پیدا بود که تحویل گیرنده باید پرشان کند. همین کار را کردم. اوراق را بردم توی اتاق و با خودنویسم عدد را روی هر سه ورق نوشتم و امضا کردم و به دست راننده دادم که راه افتاد و از همان بالا به ناظم گفتم:

- اگر مهر هم بایست زد، خودت بزن بابا.

و رفتم سراغ کارم که ناگهان در باز شد و ناظم آمد تو؛ بیجک زغال دستش بود و:

- مگه نفهمیدین آقا؟ مخصوصاً جاش رو خالی گذاشته بودند آقا...

نفهمیده بودم. اما اگر هم فهمیده بودم، فرقی نمی‌کرد و به هر صورت از چنین کودنی نا به هنگام از جا در رفتم و به شدت گفتم:

- خوب؟

- هیچ چی آقا.... رسم‌شون همینه آقا. اگه باهاشون کنار نیایید کارمونو لنگ می‌گذارند آقا...

که از جا در رفتم. به چنین صراحتی مرا که مدیر مدرسه بودم در معامله شرکت می‌داد. و فریاد زدم:

- عجب! حالا سرکار برای من تکلیف هم معین می‌کنید؟... خاک بر سر این فرهنگ با مدیرش که من باشم! برو ورقه رو بده دست‌شون، گورشون رو گم کنند. پدر سوخته‌ها...

چنان فریاد زده بودم که هیچ کس در مدرسه انتظار نداشت. مدیر سر به زیر و پا به راهی بودم که از همه خواهش می‌کردم و حالا ناظم مدرسه، داشت به من یاد می‌داد که به جای نه خروار زغال مثلا هجده خروار تحویل بگیرم و بعد با اداره‌ی فرهنگ کنار بیایم. هی هی!.... تا ظهر هیچ کاری نتوانستم بکنم، جز این‌که چند بار متن استعفانامه‌ام را بنویسم و پاره کنم... قدم اول را این جور جلوی پای آدم می‌گذارند.

بارندگی که شروع شد دستور دادم بخاری‌ها را از هفت صبح بسوزانند. بچه‌ها همیشه زود می‌آمدند. حتی روزهای بارانی. مثل این‌که اول آفتاب از خانه بیرون‌شان می‌کنند. یا ناهارنخورده. خیلی سعی کردم یک روز زودتر از بچه‌ها مدرسه باشم. اما عاقبت نشد که مدرسه را خالی از نفسِ به علم‌آلوده‌ی بچه‌ها استنشاق کنم. از راه که می‌رسیدند دور بخاری جمع می‌شدند و گیوه‌هاشان را خشک می‌کردند. و خیلی زود فهمیدم که ظهر در مدرسه ماندن هم مسأله کفش بود. هر که داشت نمی‌ماند.این قاعده در مورد معلم‌ها هم صدق می‌کرد اقلاً یک پول واکس جلو بودند. وقتی که باران می‌بارید تمام کوهپایه و بدتر از آن تمام حیاط مدرسه گل می‌شد. بازی و دویدن متوقف شده بود. مدرسه سوت و کور بود. این جا هم مسأله کفش بود. چشم اغلبشان هم قرمز بود. پیدا بود باز آن روز صبح یک فصل گریه کرده‌اند و در خانه‌شان علم صراطی بوده است.

مدرسه داشت تخته می‌شد. عده‌ی غایب‌های صبح ده برابر شده بود و ساعت اول هیچ معلمی نمی‌توانست درس بدهد. دست‌های ورم‌کرده و سرمازده کار نمی‌کرد. حتی معلم کلاس اولمان هم می‌دانست که فرهنگ و معلومات مدارس ما صرفاً تابع تمرین است. مشق و تمرین. ده بار بیست بار. دست یخ‌کرده بیل و رنده را هم نمی‌تواند به کار بگیرد که خیلی هم زمخت‌اند و دست پر کن. این بود که به فکر افتادیم. فراش جدید واردتر از همه‌ی ما بود. یک روز در اتاق دفتر، شورامانندی داشتیم که البته او هم بود. خودش را کم‌کم تحمیل کرده بود. گفت حاضر است یکی از دُم‌کلفت‌های همسایه‌ی مدرسه را وادارد که شن برایمان بفرستد به شرط آن که ما هم برویم و از انجمن محلی برای بچه‌ها کفش و لباس بخواهیم. قرار شد خودش قضیه را دنبال کند که هفته‌ی آینده جلسه‌شان کجاست و حتی بخواهد که دعوت‌مانندی از ما بکنند. دو روز بعد سه تا کامیون شن آمد. دوتایش را توی حیاط مدرسه، خالی کردیم و سومی را دم در مدرسه، و خود بچه‌ها نیم ساعته پهنش کردند. با پا و بیل و هر چه که به دست می‌رسید.

عصر همان روز ما را به انجمن دعوت کردند. خود من و ناظم باید می‌رفتیم. معلم کلاس چهارم را هم با خودمان بردیم. خانه‌ای که محل جلسه‌ی آن شب انجمن بود، درست مثل مدرسه، دور افتاده و تنها بود. قالی‌ها و کناره‌ها را به فرهنگ می‌آلودیم و می‌رفتیم. مثل این‌که سه تا سه تا روی هم انداخته بودند. اولی که کثیف شد دومی. به بالا که رسیدیم یک حاجی آقا در حال نماز خواندن بود. و صاحب‌خانه با لهجه‌ی غلیظ یزدی به استقبال‌مان آمد. همراهانم را معرفی کردم و لابد خودش فهمید مدیر کیست. برای ما چای آوردند. سیگارم را چاق کردم و با صاحب‌خانه از قالی‌هایش حرف زدیم. ناظم به بچه‌هایی می‌ماند که در مجلس بزرگترها خوابشان می‌گیرد و دل‌شان هم نمی‌خواست دست به سر شوند. سر اعضای انجمن باز شده بود. حاجی آقا صندوقدار بود. من و ناظم عین دو طفلان مسلم بودیم و معلم کلاس چهارم عین خولی وسطمان نشسته. اغلب اعضای انجمن به زبان محلی صحبت می‌کردند و رفتار ناشی داشتند. حتی یک کدامشان نمی‌دانستند که دست و پاهای خود را چه جور ضبط و ربط کنند. بلند بلند حرف می‌زدند. درست مثل این‌که وزارتخانه‌ی دواب سه تا حیوان تازه برای باغ وحش محله‌شان وارد کرده. جلسه که رسمی شد، صاحبخانه معرفی‌مان کرد و شروع کردند. مدام از خودشان صحبت می‌کردند از این‌که دزد دیشب فلان جا را گرفته و باید درخواست پاسبان شبانه کنیم و...

همین طور یک ساعت حرف زدند و به مهام امور رسیدگی کردند و من و معلم کلاس چهارم سیگار کشیدیم. انگار نه انگار که ما هم بودیم. نوکرشان که آمد استکان‌ها را جمع کند، چیزی روی جلد اشنو نوشتم و برای صاحبخانه فرستادم که یک مرتبه به صرافت ما افتاد و اجازه خواست و:

- آقایان عرضی دارند. بهتر است کارهای خودمان را بگذاریم برای بعد.

مثلاً می‌خواست بفهماند که نباید همه‌ی حرف‌ها را در حضور ما زده باشند. و اجازه دادند معلم کلاس چهار شروع کرد به نطق و او هم شروع کرد که هر چه باشد ما زیر سایه‌ی آقایانیم و خوش‌آیند نیست که بچه‌هایی باشند که نه لباس داشته باشند و نه کفش درست و حسابی و از این حرف‌ها و مدام حرف می‌زد. ناظم هم از چُرت در آمد چیزهایی را که از حفظ کرده بود گفت و التماس دعا و کار را خراب کرد.تشری به ناظم زدم که گدابازی را بگذارد کنار و حالی‌شان کردم که صحبت از تقاضا نیست و گدایی. بلکه مدرسه دور افتاده است و مستراح بی در و پیکر و از این اباطیل... چه خوب شد که عصبانی نشدم. و قرار شد که پنج نفرشان فردا عصر بیایند که مدرسه را وارسی کنند و تشکر و اظهار خوشحالی و در آمدیم.

در تاریکی بیابان هفت تا سواری پشت در خانه ردیف بودند و راننده‌ها توی یکی از آن‌ها جمع شده بودند و اسرار ارباب‌هاشان را به هم می‌گفتند. در این حین من مدام به خودم می‌گفتم من چرا رفتم؟ به من چه؟ مگر من در بی کفش و کلاهی‌شان مقصر بودم؟ می‌بینی احمق؟ مدیر مدرسه هم که باشی باید شخصیت و غرورت را لای زرورق بپیچی و طاق کلاهت بگذاری که اقلاً نپوسد. حتی اگر بخواهی یک معلم کوفتی باشی، نه چرا دور می‌زنی؟ حتی اگر یک فراش ماهی نود تومانی باشی، باید تا خرخره توی لجن فرو بروی.در همین حین که من در فکر بودم ناظم گفت:

- دیدید آقا چه طور باهامون رفتار کردند؟ با یکی از قالی‌هاشون آقا تمام مدرسه رو می‌خرید.

گفتم:

- تا سر و کارت با الف.ب است به‌پا قیاس نکنی. خودخوری می‌آره.

و معلم کلاس چهار گفت:

- اگه فحشمون هم می‌دادند من باز هم راضی بودم، باید واقع‌بین بود. خدا کنه پشیمون نشند.

بعد هم مدتی درد دل کردیم و تا اتوبوس برسد و سوار بشیم، معلوم شد که معلم کلاس چهار با زنش متارکه کرده و مادر ناظم را سرطانی تشخیص دادند. و بعد هم شب بخیر...

دو روز تمام مدرسه نرفتم. خجالت می‌کشیدم توی صورت یک کدام‌شان نگاه کنم. و در همین دو روز حاجی آقا با دو نفر آمده بودند، مدرسه را وارسی و صورت‌برداری و ناظم می‌گفت که حتی بچه‌هایی هم که کفش و کلاهی داشتند پاره و پوره آمده بودند. و برای بچه‌ها کفش و لباس خریدند. روزهای بعد احساس کردم زن‌هایی که سر راهم لب جوی آب ظرف می‌شستند، سلام می‌کنند و یک بار هم دعای خیر یکی‌شان را از عقب سر شنیدم.اما چنان از خودم بدم آمده بود که رغبتم نمی‌شد به کفش و لباس‌هاشان نگاه کنم. قربان همان گیوه‌های پاره! بله، نان گدایی فرهنگ را نو نوار کرده بود.

تازه از دردسرهای اول کار مدرسه فارغ شده بودم که شنیدم که یک روز صبح، یکی از اولیای اطفال آمد. بعد از سلام و احوالپرسی دست کرد توی جیبش و شش تا عکس در آورد، گذاشت روی میزم. شش تا عکس زن .   و هر کدام به یک حالت. یعنی چه؟ نگاه تندی به او کردم. آدم مرتبی بود. اداری مانند. کسر شأن خودم می‌دانستم که این گوشه‌ی از زندگی را طبق دستور عکاس‌باشی فلان خانه‌ی بندری ببینم. اما حالا یک مرد اتو کشیده‌ی مرتب آمده بود و شش تا از همین عکس‌ها را روی میزم پهن کرده بود و به انتظار آن که وقاحت عکس‌ها چشم‌هایم را پر کند داشت سیگار چاق می‌کرد.

حسابی غافلگیر شده بودم... حتماً تا هر شش تای عکس‌ها را ببینم، بیش از یک دقیقه طول کشید. همه از یک نفر بود. به این فکر گریختم که الان هزار ها یا میلیون ها نسخه‌ی آن، توی جیب چه جور آدم‌هایی است و در کجاها و چه قدر خوب بود که همه‌ی این آدم‌ها را می‌شناختم یا می‌دیدم. بیش ازین نمی‌شد گریخت. یارو به تمام وزنه وقاحتش، جلوی رویم نشسته بود. سیگاری آتش زدم و چشم به او دوختم. کلافه بود و پیدا بود برای کتک‌کاری هم آماده باشد. سرخ شده بود و داشت در دود سیگارش تکیه‌گاهی برای جسارتی که می‌خواست به خرج بدهد می‌جست. عکس‌ها را با یک ورقه از اباطیلی که همان روز سیاه کرده بودم، پوشاندم و بعد با لحنی که دعوا را با آن شروع می‌کنند؛ پرسیدم:

- خوب، غرض؟

و صدایم توی اتاق پیچید. حرکتی از روی بیچارگی به خودش داد و همه‌ی جسارت‌ها را با دستش توی جیبش کرد و آرام‌تر از آن چیزی که با خودش تو آورده بود، گفت:

- چه عرض کنم؟... از معلم کلاس پنج تون بپرسید.

که راحت شدم و او شروع کرد به این که «این چه فرهنگی است؟ خراب بشود. پس بچه‌های مردم با چه اطمینانی به مدرسه بیایند؟

و از این حرف‌ها...

خلاصه این آقا معلم کاردستی کلاس پنجم، این عکس‌ها را داده به پسر آقا تا آن‌ها را روی تخته سه لایی بچسباند و دورش را سمباده بکشد و بیاورد. به هر صورت معلم کلاس پنج بی‌گدار به آب زده. و حالا من چه بکنم؟ به او چه جوابی بدهم؟ بگویم معلم را اخراج می‌کنم؟ که نه می‌توانم و نه لزومی دارد. او چه بکند؟ حتماً در این شهر کسی را ندارد که به این عکس‌ها دلخوش کرده. ولی آخر چرا این جور؟ یعنی این قدر احمق است که حتی شاگردهایش را نمی‌شناسد؟... پاشدم ناظم را صدا بزنم که خودش آمده بود بالا، توی ایوان منتظر ایستاده بود. من آخرین کسی بودم که از هر اتفاقی در مدرسه خبردار می‌شدم. حضور این ولی طفل گیجم کرده بود که چنین عکس‌هایی را از توی جیب پسرش، و لابد به همین وقاحتی که آن‌ها را روی میز من ریخت، در آورده بوده. وقتی فهمید هر دو در مانده‌ایم سوار بر اسب شد که اله می‌کنم و بله می‌کنم، در مدرسه را می‌بندم، و از این جفنگیات....

حتماً نمی‌دانست که اگر در هر مدرسه بسته بشود، در یک اداره بسته شده است. اما من تا او بود نمی‌توانستم فکرم را جمع کنم. می‌خواست پسرش را بخواهیم تا شهادت بدهد و چه جانی کندیم تا حالیش کنیم که پسرش هر چه خفت کشیده، بس است و وعده‌ها دادیم که معلمش را دم خورشید کباب کنیم و از نان خوردن بیندازیم. یعنی اول ناظم شروع کرد که از دست او دل پری داشت و من هم دنبالش را گرفتم. برای دک کردن او چاره‌ای جز این نبود. و بعد رفت، ما دو نفری ماندیم با شش تا عکس زن . حواسم که جمع شد به ناظم سپردم صدایش را در نیاورد و یک هفته‌ی تمام مطلب را با عکس‌ها، توی کشوی میزم قفل کردم و بعد پسرک را صدا زدم. نه عزیزدُردانه می‌نمود و نه هیچ جور دیگر. داد می‌زد که از خانواده‌ی عیال‌واری است. کم‌خونی و فقر. دیدم معلمش زیاد هم بد تشخیص نداده. یعنی زیاد بی‌گدار به آب نزده. گفتم:

- خواهر برادر هم داری؟

- آ... آ...آقا داریم آقا.

- چند تا؟

- آ... آقا چهار تا آقا.

- عکس‌ها رو خودت به بابات نشون دادی؟

- نه به خدا آقا... به خدا قسم...

- پس چه طور شد؟

و دیدم از ترس دارد قالب تهی می‌کند. گرچه چوب‌های ناظم شکسته بود، اما ترس او از من که مدیر باشم و از ناظم و از مدرسه و از تنبیه سالم مانده بود.

- نترس بابا. کاریت نداریم. تقصیر آقا معلمه که عکس‌ها رو داده... تو کار بدی نکردی بابا جان. فهمیدی؟ اما می‌خواهم ببینم چه طور شد که عکس‌ها دست بابات افتاد.

- آ.. آ... آخه آقا... آخه...

می‌دانستم که باید کمکش کنم تا به حرف بیاید.

گفتم:

- می‌دونی بابا؟ عکس‌هام چیز بدی نبود. تو خودت فهمیدی چی بود؟

- آخه آقا...نه آقا.... خواهرم آقا... خواهرم می‌گفت...

- خواهرت؟ از تو کوچک‌تره؟

- نه آقا. بزرگ‌تره. می‌گفتش که آقا... می‌گفتش که آقا... هیچ چی سر عکس‌ها دعوامون شد.

دیگر تمام بود. عکس‌ها را به خواهرش نشان داده بود که لای دفترچه پر بوده از عکس آرتیست‌ها. به او پز داده بوده. اما حاضر نبوده، حتی یکی از آن‌ها را به خواهرش بدهد. آدم مورد اعتماد معلم باشد و چنین خبطی بکند؟ و تازه جواب معلم را چه بدهد؟ ناچار خواهر او را لو داده بوده. بعد از او معلم را احضار کردم. علت احضار را می‌دانست. و داد می‌زد که چیزی ندارد بگوید. پس از یک هفته مهلت، هنوز از وقاحتی که من پیدا کرده بودم، تا از آدم خلع سلاح‌شده‌ای مثل او، دست بر ندارم، در تعجب بود. به او سیگار تعارف کردم و این قصه را برایش تعریف کردم که در اوایل تأسیس وزارت معارف، یک روز به وزیر خبر می‌دهند که فلان معلم با فلان بچه روابطی دارد. وزیر فوراً او را می‌خواهد و حال و احوال او را می‌پرسد و این‌که چرا تا به حال زن نگرفته و ناچار تقصیر گردن بی‌پولی می‌افتد و دستور که فلان قدر به او کمک کنند تا عروسی راه بیندازد و خود او هم دعوت بشود و قضیه به همین سادگی تمام می‌شود. و بعد گفتم که خیلی جوان‌ها هستند که نمی‌توانند زن بگیرند و وزرای فرهنگ هم این روزها گرفتار مصاحبه‌های روزنامه‌ای و رادیویی هستند. اما در نجیب‌خانه‌ها که باز است و ازین مزخرفات... و هم‌دردی و نگذاشتم یک کلمه حرف بزند. بعد هم عکس را که توی پاکت گذاشته بودم، به دستش دادم و وقاحت را با این جمله به حد اعلا رساندم که:

- اگر به تخته نچسبونید، ضررشون کم‌تره.

تا حقوقم به لیست اداره‌ی فرهنگ برسه، سه ماه طول کشید. فرهنگی‌های گداگشنه و خزانه‌ی خالی و دست‌های از پا درازتر! اما خوبیش این بود که در مدرسه‌ی ما فراش جدیدمان پولدار بود و به همه‌شان قرض داد. کم کم بانک مدرسه شده بود. از سیصد و خرده‌ای تومان که می‌گرفت، پنجاه تومان را هم خرج نمی‌کرد. نه سیگار می‌کشید و نه اهل سینما بود و نه برج دیگری داشت. از این گذشته، باغبان یکی از دم‌کلفت‌های همان اطراف بود و باغی و دستگاهی و سور و ساتی و لابد آشپزخانه‌ی مرتبی. خیلی زود معلم‌ها فهمیدند که یک فراش پولدار خیلی بیش‌تر به درد می‌خورد تا یک مدیر بی‌بو و خاصیت.

این از معلم‌ها. حقوق مرا هم هنوز از مرکز می‌دادند. با حقوق ماه بعد هم اسم مرا هم به لیست اداره منتقل کردند. درین مدت خودم برای خودم ورقه انجام کار می‌نوشتم و امضا می‌کردم و می‌رفتم از مدرسه‌ای که قبلاً در آن درس می‌دادم، حقوقم را می‌گرفتم. سر و صدای حقوق که بلند می‌شد معلم‌ها مرتب می‌شدند و کلاس ماهی سه چهار روز کاملاً دایر بود. تا ورقه‌ی انجام کار به دستشان بدهم. غیر از همان یک بار - در اوایل کار- که برای معلم حساب پنج و شش قرمز توی دفتر گذاشتیم، دیگر با مداد قرمز کاری نداشتیم و خیال همه‌شان راحت بود. وقتی برای گرفتن حقوقم به اداره رفتم، چنان شلوغی بود که به خودم گفتم کاش اصلاً حقوقم را منتقل نکرده بودم. نه می‌توانستم سر صف بایستم و نه می‌توانستم از حقوقم بگذرم. تازه مگر مواجب‌بگیر دولت چیزی جز یک انبان گشاده‌ی پای صندوق است؟..... و اگر هم می‌ماندی با آن شلوغی باید تا دو بعداز ظهر سر پا بایستی. همه‌ی جیره‌خوارهای اداره بو برده بودند که مدیرم. و لابد آن‌قدر ساده لوح بودند که فکر کنند روزی گذارشان به مدرسه‌ی ما بیفتد. دنبال سفته‌ها می‌گشتند، به حسابدار قبلی فحش می‌دادند، التماس می‌کردند که این ماه را ندیده بگیرید و همه‌ی حق و حساب‌دان شده بودند و یکی که زودتر از نوبت پولش را می‌گرفت صدای همه در می‌آمد. در لیست مدرسه، بزرگ‌ترین رقم مال من بود. درست مثل بزرگ‌ترین گناه در نامه‌ی عمل. دو برابر فراش جدیدمان حقوق می‌گرفتم. از دیدن رقم‌های مردنی حقوق دیگران چنان خجالت کشیدم که انگار مال آن‌ها را دزدیده‌ام. و تازه خلوت که شد و ده پانزده تا امضا که کردم، صندوق‌دار چشمش به من افتاد و با یک معذرت، شش صد تومان پول دزدی را گذاشت کف دستم... مرده شور!

هنوز برف اول نباریده بود که یک روز عصر، معلم کلاس چهار رفت زیر ماشین. زیر یک سواری. مثل همه‌ی عصرها من مدرسه نبودم. دم غروب بود که فراش قدیمی مدرسه دم در خونه‌مون، خبرش را آورد. که دویدم به طرف لباسم و تا حاضر بشوم، می‌شنیدم که دارد قضیه را برای زنم تعریف می‌کند. ماشین برای یکی از آمریکایی‌ها بوده. باقیش را از خانه که در آمدیم برایم تعریف کرد. گویا یارو خودش پشت فرمون بوده و بعد هم هول شده و در رفته. بچه‌ها خبر را به مدرسه برگردانده‌اند و تا فراش و زنش برسند، جمعیت و پاسبان‌ها سوارش کرده بودند و فرستاده بوده‌اند مریض‌خانه. به اتوبوس که رسیدم، دیدم لاک پشت است. فراش را مرخص کردم و پریدم توی تاکسی. اول رفتم سراغ پاسگاه جدید کلانتری. تعاریف تکه و پاره‌ای از پرونده مطلع بود. اما پرونده تصریحی نداشت که راننده که بوده. اما هیچ کس نمی‌دانست عاقبت چه بلایی بر سر معلم کلاس چهار ما آمده است. کشیک پاسگاه همین قدر مطلع بود که درین جور موارد «طبق جریان اداری» اول می‌روند سرکلانتری، بعد دایره‌ی تصادفات و بعد بیمارستان. اگر آشنا در نمی‌آمدیم، کشیک پاسگاه مسلماً نمی‌گذاشت به پرونده نگاه چپ بکنم. احساس کردم میان اهل محل کم‌کم دارم سرشناس می‌شوم. و از این احساس خنده‌ام گرفت.

ساعت ۸ دم در بیمارستان بودم، اگر سالم هم بود حتماً یه چیزیش شده بود. همان طور که من یه چیزیم می‌شد. روی در بیمارستان نوشته شده بود: «از ساعت ۷ به بعد ورود ممنوع». در زدم. از پشت در کسی همین آیه را صادر کرد. دیدم فایده ندارد و باید از یک چیزی کمک بگیرم. از قدرتی، از مقامی، از هیکلی، از یک چیزی. صدایم را کلفت کردم و گفتم:« من...» می‌خواستم بگویم من مدیر مدرسه‌ام. ولی فوراً پشیمان شدم. یارو لابد می‌گفت مدیر مدرسه کدام سگی است؟ این بود با کمی مکث و طمطراق فراوان جمله‌ام را این طور تمام کردم:

- ...بازرس وزارت فرهنگم.

که کلون صدایی کرد و لای در باز شد. یارو با چشم‌هایش سلام کرد. رفتم تو و با همان صدا پرسیدم:

- این معلمه مدرسه که تصادف کرده...

تا آخرش را خواند. یکی را صدا زد و دنبالم فرستاد که طبقه‌ی فلان، اتاق فلان. از حیاط به راهرو و باز به حیاط دیگر که نصفش را برف پوشانده بود و من چنان می‌دویدم که یارو از عقب سرم هن هن می‌کرد. طبقه‌ی اول و دوم و چهارم. چهار تا پله یکی. راهرو تاریک بود و پر از بوهای مخصوص بود. هن هن کنان دری را نشان داد که هل دادم و رفتم تو. بو تندتر بود و تاریکی بیشتر. تالاری بود پر از تخت و جیرجیر کفش و خرخر یک نفر. دور یک تخت چهار نفر ایستاده بودند. حتماً خودش بود. پای تخت که رسیدم، احساس کردم همه‌ی آنچه از خشونت و تظاهر و ابهت به کمک خواسته بودم آب شد و بر سر و صورتم راه افتاد. و این معلم کلاس چهارم مدرسه‌ام بود. سنگین و با شکم بر آمده دراز کشیده بود. خیلی کوتاه‌تر از زمانی که سر پا بود به نظرم آمد. صورت و سینه‌اش از روپوش چرک‌مُرد بیرون بود. صورتش را که شسته بودند کبود کبود بود، درست به رنگ جای سیلی روی صورت بچه‌ها. مرا که دید، لبخند و چه لبخندی! شاید می‌خواست بگوید مدرسه‌ای که مدیرش عصرها سر کار نباشد، باید همین جورها هم باشد. خنده توی صورت او همین طور لرزید و لرزید تا یخ زد.

«آخر چرا تصادف کردی؟...»

مثل این که سوال را ازو کردم. اما وقتی که دیدم نمی‌تواند حرف بزند و به جای هر جوابی همان خنده‌ی یخ‌بسته را روی صورت دارد، خودم را به عنوان او دم چک گرفتم. «آخه چرا؟ چرا این هیکل مدیر کلی را با خودت این قد این ور و آن ور می‌بری تا بزنندت؟ تا زیرت کنند؟ مگر نمی‌دانستی که معلم حق ندارد این قدر خوش‌هیکل باشد؟ آخر چرا تصادف کردی؟» به چنان عتاب و خطابی این‌ها را می‌گفتم که هیچ مطمئن نیستم بلند بلند به خودش نگفته باشم. و یک مرتبه به کله‌ام زد که «مبادا خودت چشمش زده باشی؟» و بعد: «احمق خاک بر سر! بعد از سی و چند سال عمر، تازه خرافاتی شدی!» و چنان از خودم بیزاریم گرفت که می‌خواستم به یکی فحش بدهم، کسی را بزنم. که چشمم به دکتر کشیک افتاد.

- مرده شور این مملکتو ببره. ساعت چهار تا حالا از تن این مرد خون می‌ره. حیفتون نیومد؟...

دستی روی شانه‌ام نشست و فریادم را خواباند. برگشتم پدرش بود. او هم می‌خندید. دو نفر دیگر هم با او بودند. همه دهاتی‌وار؛ همه خوش قد و قواره. حظ کردم! آن دو تا پسرهایش بودند یا برادرزاده‌هایش یا کسان دیگرش. تازه داشت گل از گلم می‌شکفت که شنیدم:

- آقا کی باشند؟

این راهم دکتر کشیک گفت که من باز سوار شدم:

- مرا می‌گید آقا؟ من هیشکی. یک آقا مدیر کوفتی. این هم معلمم.

که یک مرتبه عقل هی زد و «پسر خفه شو» و خفه شدم. بغض توی گلویم بود. دلم می‌خواست یک کلمه دیگر بگوید. یک کنایه بزند... نسبت به مهارت هیچ دکتری تا کنون نتوانسته‌ام قسم بخورم. دستش را دراز کرد که به اکراه فشار دادم و بعد شیشه‌ی بزرگی را نشانم داد که وارونه بالای تخت آویزان بود و خرفهمم کرد که این جوری غذا به او می‌رسانند و عکس هم گرفته‌اند و تا فردا صبح اگر زخم‌ها چرک نکند، جا خواهند انداخت و گچ خواهند کرد. که یکی دیگر از راه رسید. گوشی به دست و سفید پوش و معطر. با حرکاتی مثل آرتیست سینما. سلامم کرد. صدایش در ته ذهنم چیزی را مختصر تکانی داد. اما احتیاجی به کنجکاوی نبود. یکی از شاگردهای نمی‌دانم چند سال پیشم بود. خودش خودش را معرفی کرد. آقای دکتر...! عجب روزگاری! هر تکه از وجودت را با مزخرفی از انبان مزخرفاتت، مثل ذره‌ای روزی در خاکی ریخته‌ای که حالا سبز کرده. چشم داری احمق. این تویی که روی تخت دراز کشیده‌ای. ده سال آزگار از پلکان ساعات و دقایق عمرت هر لحظه یکی بالا رفته و تو فقط خستگی این بار را هنوز در تن داری. این جوجه‌فکلی و جوجه‌های دیگر که نمی‌شناسی‌شان، همه از تخمی سر در آورده‌اند که روزی حصار جوانی تو بوده و حالا شکسته و خالی مانده. دستش را گرفتم و کشیدمش کناری و در گوشش هر چه بد و بی‌راه می‌دانستم، به او و همکارش و شغلش دادم. مثلاً می‌خواستم سفارش معلم کلاس چهار مدرسه‌ام را کرده باشم. بعد هم سری برای پدر تکان دادم و گریختم. از در که بیرون آمدم، حیاط بود و هوای بارانی. از در بزرگ که بیرون آمدم به این فکر می‌کردم که «اصلا به تو چه؟ اصلاً چرا آمدی؟ می‌خواستی کنجکاوی‌ات را سیرکنی؟» و دست آخر به این نتیجه رسیدم که «طعمه‌ای برای میزنشین‌های شهربانی و دادگستری به دست آمده و تو نه می‌توانی این طعمه را از دستشان بیرون بیاوری و نه هیچ کار دیگری می‌توانی بکنی...»

و داشتم سوار تاکسی می‌شدم تا برگردم خانه که یک دفعه به صرافت افتادم که اقلاً چرا نپرسیدی چه بلایی به سرش آمده؟» خواستم عقب‌گرد کنم، اما هیکل کبود معلم کلاس چهارم روی تخت بود و دیدم نمی‌توانم. خجالت می‌کشیدم و یا می‌ترسیدم. آن شب تا ساعت دو بیدار بودم و فردا یک گزارش مفصل به امضای مدیر مدرسه و شهادت همه‌ی معلم‌ها برای اداره‌ی فرهنگ و کلانتری محل و بعد هم دوندگی در اداره‌ی بیمه و قرار بر این که روزی نه تومان بودجه برای خرج بیمارستان او بدهند و عصر پس از مدتی رفتم مدرسه و کلاس‌ها را تعطیل کردم و معلم‌ها و بچه‌های ششم را فرستادم عیادتش و دسته گل و ازین بازی‌ها... و یک ساعتی در مدرسه تنها ماندم و فارغ از همه چیز برای خودم خیال بافتم.... و فردا صبح پدرش آمد سلام و احوالپرسی و گفت یک دست و یک پایش شکسته و کمی خونریزی داخل مغز و از طرف یارو آمریکاییه آمده‌اند عیادتش و وعده و وعید که وقتی خوب شد، در اصل چهار استخدامش کنند و با زبان بی‌زبانی حالیم کرد که گزارش را بیخود داده‌ام و حالا هم داده‌ام، دنبالش نکنم و رضایت طرفین و کاسه‌ی از آش داغ‌تر و از این حرف‌ها... خاک بر سر مملکت.

اوایل امر توجهی به بچه‌ها نداشتم. خیال می‌کردم اختلاف سِنی میان‌مان آن قدر هست که کاری به کار همدیگر نداشته باشیم. همیشه سرم به کار خودم بود. در دفتر را می‌بستم و در گرمای بخاری دولت قلم صد تا یک غاز می‌زدم. اما این کار مرتب سه چهار هفته بیش‌تر دوام نکرد. خسته شدم. ناچار به مدرسه بیشتر می‌رسیدم. یاد روزهای قدیمی با دوستان قدیمی به خیر چه آدم‌های پاک و بی‌آلایشی بودند، چه شخصیت‌های بی‌نام و نشانی و هر کدام با چه زبانی و با چه ادا و اطوارهای مخصوص به خودشان و این جوان‌های چلفته‌ای. چه مقلدهای بی‌دردسری برای فرهنگی‌مابی! نه خبری از دیروزشان داشتند و نه از املاک تازه‌ای که با هفتاد واسطه به دست‌شان داده بودند، چیزی سرشان می‌شد. بدتر از همه بی‌دست و پایی‌شان بود. آرام و مرتب درست مثل واگن شاه عبدالعظیم می‌آمدند و می‌رفتند. فقط بلد بودند روزی ده دقیقه دیرتر بیایند و همین. و از این هم بدتر تنگ‌نظری‌شان بود.

سه بار شاهد دعواهایی بودم که سر یک گلدان میخک یا شمعدانی بود. بچه‌باغبان‌ها زیاد بودند و هر کدام‌شان حداقل ماهی یک گلدان میخک یا شمعدانی می‌آوردند که در آن برف و سرما نعمتی بود. اول تصمیم گرفتم، مدرسه را با آن‌ها زینت دهم. ولی چه فایده؟ نه کسی آب‌شان می‌داد و نه مواظبتی. و باز بدتر از همه‌ی این‌ها، بی‌شخصیتی معلم‌ها بود که درمانده‌ام کرده بود. دو کلمه نمی‌توانستند حرف بزنند. عجب هیچ‌کاره‌هایی بودند! احساس کردم که روز به روز در کلاس‌ها معلم‌ها به جای دانش‌آموزان جاافتاده‌تر می‌شوند. در نتیجه گفتم بیش‌تر متوجه بچه‌ها باشم.

آن‌ها که تنها با ناظم سر و کار داشتند و مثل این بود که به من فقط یک سلام نیمه‌جویده بدهکارند. با این همه نومیدکننده نبودند. توی کوچه مواظب‌شان بودم. می‌خواستم حرف و سخن‌ها و درد دل‌ها و افکارشان را از یک فحش نیمه‌کاره یا از یک ادای نیمه‌تمام حدس بزنم، که سلام‌نکرده در می‌رفتند. خیلی کم تنها به مدرسه می‌آمدند. پیدا بود که سر راه همدیگر می‌ایستند یا در خانه‌ی یکدیگر می‌روند. سه چهار نفرشان هم با اسکورت می‌آمدند. از بیست سی نفری که ناهار می‌ماندند، فقط دو نفرشان چلو خورش می‌آوردند؛ فراش اولی مدرسه برایم خبر می‌آورد. بقیه گوشت‌کوبیده، پنیر گردوئی، دم پختکی و از این جور چیزها. دو نفرشان هم بودند که نان سنگک خالی می‌آوردند. برادر بودند. پنجم و سوم. صبح که می‌آمدند، جیب‌هاشان باد کرده بود. سنگک را نصف می‌کردند و توی جیب‌هاشان می‌تپاندند و ظهر می‌شد، مثل آن‌هایی که ناهارشان را در خانه می‌خورند، می‌رفتند بیرون. من فقط بیرون رفتن‌شان را می‌دیدم. اما حتی همین‌ها هر کدام روزی، یکی دو قران از فراش مدرسه خرت و خورت می‌خریدند. از همان فراش قدیمی مدرسه که ماهی پنج تومان سرایداریش را وصول کرده بودم. هر روز که وارد اتاقم می‌شدم پشت سر من می‌آمد بارانی‌ام را بر می‌داشت و شروع می‌کرد به گزارش دادن، که دیروز باز دو نفر از معلم‌ها سر یک گلدان دعوا کرده‌اند یا مأمور فرماندار نظامی آمده یا دفتردار عوض شده و از این اباطیل... پیدا بود که فراش جدید هم در مطالبی که او می‌گفت، سهمی دارد.

یک روز در حین گزارش دادن، اشاره‌ای کرد به این مطلب که دیروز عصر یکی از بچه‌های کلاس چهار دو تا کله قند به او فروخته است. درست مثل اینکه سر کلاف را به دستم داده باشد پرسیدم:

- چند؟

- دو تومنش دادم آقا.

- زحمت کشیدی. نگفتی از کجا آورده؟

- من که ضامن بهشت و جهنمش نبودم آقا.

بعد پرسیدم:

- چرا به آقای ناظم خبر ندادی؟

می‌دانستم که هم او و هم فراش جدید، ناظم را هووی خودشان می‌دانند و خیلی چیزهاشان از او مخفی بود. این بود که میان من و ناظم خاصه‌خرجی می‌کردند. در جوابم همین طور مردد مانده بود که در باز شد و فراش جدید آمد تو. که:

- اگه خبرش می‌کرد آقا بایست سهمش رو می‌داد...

اخمم را درهم کشیدم و گفتم:

- تو باز رفتی تو کوک مردم! اونم این جوری سر نزده که نمی‌آیند تو اتاق کسی، پیرمرد!

و بعد اسم پسرک را ازشان پرسیدم و حالی‌شان کردم که چندان مهم نیست و فرستادمشان برایم چای بیاورند. بعد کارم را زودتر تمام کردم و رفتم به اتاق دفتر احوالی از مادر ناظم پرسیدم و به هوای ورق زدن پرونده‌ها فهمیدم که پسرک شاگرد دوساله است و پدرش تاجر بازار. بعد برگشتم به اتاقم. یادداشتی برای پدر نوشتم که پس فردا صبح، بیاید مدرسه و دادم دست فراش جدید که خودش برساند و رسیدش را بیاورد.

و پس فردا صبح یارو آمد. باید مدیر مدرسه بود تا دانست که اولیای اطفال چه راحت تن به کوچک‌ترین خرده‌فرمایش‌های مدرسه می‌دهند. حتم دارم که اگر از اجرای ثبت هم دنبال‌شان بفرستی به این زودی‌ها آفتابی نشوند. چهل و پنج ساله مردی بود با یخه‌ی بسته بی‌کراوات و پالتویی که بیش‌تر به قبا می‌ماند. و خجالتی می‌نمود. هنوز ننشسته، پرسیدم:

- شما دو تا زن دارید آقا؟

درباره‌ی پسرش برای خودم پیش‌گویی‌هایی کرده بودم و گفتم این طوری به او رودست می‌زنم. پیدا بود که از سؤالم زیاد یکه نخورده است. گفتم برایش چای آوردند و سیگاری تعارفش کردم که ناشیانه دود کرد از ترس این که مبادا جلویم در بیاید که - به شما چه مربوط است و از این اعتراض‌ها - امانش ندادم و سؤالم را این جور دنبال کردم:

- البته می‌بخشید. چون لابد به همین علت بچه شما دو سال در یک کلاس مانده.

شروع کرده بودم برایش یک میتینگ بدهم که پرید وسط حرفم:

- به سر شما قسم، روزی چهار زار پول تو جیبی داره آقا. پدرسوخته‌ی نمک به حروم...!

حالیش کردم که علت، پول تو جیبی نیست و خواستم که عصبانی نشود و قول گرفتم که اصلاً به روی پسرش هم نیاورد و آن وقت میتینگم را برایش دادم که لابد پسر در خانه مهر و محبتی نمی‌بیند و غیب‌گویی‌های دیگر... تا عاقبت یارو خجالتش ریخت و سرِ درد دلش باز شد که عفریته زن اولش همچه بوده و همچون بوده و پسرش هم به خودش برده و کی طلاقش داده و از زن دومش چند تا بچه دارد و این نره‌خر حالا باید برای خودش نان‌آور شده باشد و زنش حق دارد که با دو تا بچه‌ی خرده‌پا به او نرسد... من هم کلی برایش صحبت کردم. چایی دومش را هم سر کشید و قول‌هایش را که داد و رفت، من به این فکر افتادم که «نکند علمای تعلیم و تربیت هم، همین جورها تخم دوزرده می‌کنند!»

یک روز صبح که رسیدم، ناظم هنوز نیامده بود. از این اتفاق‌ها کم می‌افتاد. ده دقیقه‌ای از زنگ می‌گذشت و معلم‌ها در دفتر سرگرم اختلاط بودند. خودم هم وقتی معلم بودم به این مرض دچار بودم. اما وقتی مدیر شدم تازه فهمیدم که معلم‌ها چه لذتی می‌برند. حق هم داشتند. آدم وقتی مجبور باشد شکلکی را به صورت بگذارد که نه دیگران از آن می‌خندند و نه خود آدم لذتی می‌برد، پیداست که رفع تکلیف می‌کند. زنگ را گفتم زدند و بچه‌ها سر کلاس رفتند. دو تا از کلاس‌ها بی‌معلم بود. یکی از ششمی‌ها را فرستادم سر کلاس سوم که برای‌شان دیکته بگوید و خودم رفتم سر کلاس چهار. مدیر هم که باشی، باز باید تمرین کنی که مبادا فوت و فن معلمی از یادت برود. در حال صحبت با بچه‌ها بودم که فراش خبر آورد که خانمی توی دفتر منتظرم است. خیال کردم لابد همان زنکه‌ی بیکاره‌ای است که هفته‌ای یک بار به هوای سرکشی، به وضع درس و مشق بچه‌اش سری می‌زند. زن سفیدرویی بود با چشم‌های درشت محزون و موی بور. بیست و پنج ساله هم نمی‌نمود. اما بچه‌اش کلاس سوم بود. روز اول که دیدمش لباس نارنجی به تن داشت و تن بزک کرده بود. از زیارت من خیلی خوشحال شد و از مراتب فضل و ادبم خبر داشت.

خیلی ساده آمده بود تا با دو تا مرد حرفی زده باشد. آن طور که ناظم خبر می‌داد، یک سالی طلاق گرفته بود و روی هم رفته آمد و رفتنش به مدرسه باعث دردسر بود. وسط بیابان و مدرسه‌ای پر از معلم‌های عزب و بی‌دست و پا و یک زن زیبا... ناچار جور در نمی‌آمد. این بود که دفعات بعد دست به سرش می‌کردم، اما از رو نمی‌رفت. سراغ ناظم و اتاق دفتر را می‌گرفت و صبر می‌کرد تا زنگ را بزنند و معلم‌ها جمع بشوند و لابد حرف و سخنی و خنده‌ای و بعد از معلم کلاس سوم سراغ کار و بار و بچه‌اش را می‌گرفت و زنگ بعد را که می‌زدند، خداحافظی می‌کرد و می‌رفت. آزاری نداشت. با چشم‌هایش نفس معلم‌ها را می‌برید. و حالا باز هم همان زن بود و آمده بود و من تا از پلکان پایین بروم در ذهنم جملات زننده‌ای ردیف می‌کردم، تا پایش را از مدرسه ببرد که در را باز کردم و سلام...

عجب! او نبود. دخترک یکی دو ساله‌ای بود با دهان گشاد و موهای زبرش را به زحمت عقب سرش گلوله کرده بود و بفهمی نفهمی دستی توی صورتش برده بود. روی هم رفته زشت نبود. اما داد می‌زد که معلم است. گفتم که مدیر مدرسه‌ام و حکمش را داد دستم که دانشسرا دیده بود و تازه استخدام شده بود. برایمان معلم فرستاده بودند. خواستم بگویم «مگر رئیس فرهنگ نمی‌داند که این جا بیش از حد مرد است» ولی دیدم لزومی ندارد و فکر کردم این هم خودش تنوعی است.

به هر صورت زنی بود و می‌توانست محیط خشن مدرسه را که به طرز ناشیانه‌ای پسرانه بود، لطافتی بدهد و خوش‌آمد گفتم و چای آوردند که نخورد و بردمش کلاس‌های سوم و چهارم را نشانش دادم که هر کدام را مایل است، قبول کند و صحبت از هجده ساعت درس که در انتظار او بود و برگشتیم به دفتر .پرسید غیر از او هم، معلم زن داریم. گفتم:

- متأسفانه راه مدرسه‌ی ما را برای پاشنه‌ی کفش خانم‌ها نساخته‌اند.

که خندید و احساس کردم زورکی می‌خندد. بعد کمی این دست و آن دست کرد و عاقبت:

- آخه من شنیده بودم شما با معلماتون خیلی خوب تا می‌کنید.

صدای جذابی داشت. فکر کردم حیف که این صدا را پای تخته سیاه خراب خواهد کرد. و گفتم:

- اما نه این قدر که مدرسه تعطیل بشود خانم! و لابد به عرض‌تون رسیده که همکارهای شما، خودشون نشسته‌اند و تصمیم گرفته‌اند که هجده ساعت درس بدهند. بنده هیچ‌کاره‌ام.

- اختیار دارید.

و نفهمیدم با این «اختیار دارید» چه می‌خواست بگوید. اما پیدا بود که بحث سر ساعات درس نیست. آناً تصمیم گرفتم، امتحانی بکنم:

- این را هم اطلاع داشته باشید که فقط دو تا از معلم‌های ما متأهل‌اند.

که قرمز شد و برای این که کار دیگری نکرده باشد، برخاست و حکمش را از روی میز برداشت. پا به پا می‌شد که دیدم باید به دادش برسم. ساعت را از او پرسیدم. وقت زنگ بود. فراش را صدا کردم که زنگ را بزند و بعد به او گفتم، بهتر است مشورت دیگری هم با رئیس فرهنگ بکند و ما به هر صورت خوشحال خواهیم شد که افتخار همکاری با خانمی مثل ایشان را داشته باشیم و خداحافظ شما. از در دفتر که بیرون رفت، صدای زنگ برخاست و معلم‌ها انگار موشان را آتش زده‌اند، به عجله رسیدند و هر کدام از پشت سر، آن قدر او را پاییدند تا از در بزرگ آهنی مدرسه بیرون رفت.

فردا صبح معلوم شد که ناظم، دنبال کار مادرش بوده است که قرار بود بستری شود، تا جای سرطان گرفته را یک دوره برق بگذارند. کل کار بیمارستان را من به کمک دوستانم انجام دادم و موقع آن رسیده بود که مادرش برود بیمارستان اما وحشتش گرفته بود و حاضر نبود به بیمارستان برود. و ناظم می‌خواست رسماً دخالت کنم و با هم برویم خانه‌شان و با زبان چرب و نرمی که به قول ناظم داشتم مادرش را راضی کنم. چاره‌ای نبود. مدرسه را به معلم‌ها سپردیم و راه افتادیم. بالاخره به خانه‌ی آن‌ها رسیدیم. خانه‌ای بسیار کوچک و اجاره‌ای. مادر با چشم‌های گود نشسته و انگار زغال به صورت مالیده! سیاه نبود اما رنگش چنان تیره بود که وحشتم گرفت. اصلاً صورت نبود. زخم سیاه شده‌ای بود که انگار از جای چشم‌ها و دهان سر باز کرده است. کلی با مادرش صحبت کردم. از پسرش و کلی دروغ و دونگ، و چادرش را روی چارقدش انداختیم و علی... و خلاصه در بیمارستان بستری شدند.

فردا که به مدرسه آمدم، ناظم سرحال بود و پیدا بود که از شر چیزی خلاص شده است و خبر داد که معلم کلاس سه را گرفته‌اند. یک ماه و خرده‌ای می‌شد که مخفی بود و ما ورقه‌ی انجام کارش را به جانشین غیر رسمی‌اش داده بودیم و حقوقش لنگ نشده بود و تا خبر رسمی بشنود و در روزنامه‌ای بیابد و قضیه به اداره‌ی فرهنگ و لیست حقوق بکشد، باز هم می‌دادیم. اما خبر که رسمی شد، جانشین واجد شرایط هم نمی‌توانست بفرستد و باید طبق مقررات رفتار می‌کردیم و بدیش همین بود. کم کم احساس کردم که مدرسه خلوت شده است و کلاس‌ها اغلب اوقات بی‌کارند. جانشین معلم کلاس چهار هنوز سر و صورتی به کارش نداده بود و حالا یک کلاس دیگر هم بی‌معلم شد. این بود که باز هم به سراغ رئیس فرهنگ رفتم. معلوم شد آن دخترک ترسیده و «نرسیده متلک پیچش کرده‌اید» رئیس فرهنگ این طور می‌گفت. و ترجیح داده بود همان زیر نظر خودش دفترداری کند. و بعد قول و قرار و فردا و پس فردا و عاقبت چهار روز دوندگی تا دو تا معلم گرفتم. یکی جوانکی رشتی که گذاشتیمش کلاس چهار و دیگری باز یکی ازین آقاپسرهای بریانتین‌زده که هر روز کراوات عوض می‌کرد، با نقش‌ها و طرح‌های عجیب. عجب فرهنگ را با قرتی‌ها در آمیخته بودند! باداباد. او را هم گذاشتیم سر کلاس سه. اواخر بهمن، یک روز ناظم آمد اتاقم که بودجه‌ی مدرسه را زنده کرده است. گفتم:

- مبارکه، چه قدر گرفتی؟

- هنوز هیچ چی آقا. قراره فردا سر ظهر بیاند این جا آقا و همین جا قالش رو بکنند.

و فردا اصلاً مدرسه نرفتم. حتماً می‌خواست من هم باشم و در بده بستان ماهی پانزده قران، حق نظافت هر اتاق نظارت کنم و از مدیریتم مایه بگذارم تا تنخواه‌گردان مدرسه و حق آب و دیگر پول‌های عقب‌افتاده وصول بشود... فردا سه نفری آمده بودند مدرسه. ناهار هم به خرج ناظم خورده بودند. و قرار دیگری برای یک سور حسابی گذاشته بودند و رفته بودند و ناظم با زبان بی‌زبانی حالیم کرد که این بار حتماً باید باشم و آن طور که می‌گفت، جای شکرش باقی بود که مراعات کرده بودند و حق بوقی نخواسته بودند. اولین باری بود که چنین اهمیتی پیدا می‌کردم. این هم یک مزیت دیگر مدیری مدرسه بود! سی صد تومان از بودجه‌ی دولت بسته به این بود که به فلان مجلس بروی یا نروی. تا سه روز دیگر موعد سور بود، اصلاً یادم نیست چه کردم. اما همه‌اش در این فکر بودم که بروم یا نروم؟ یک بار دیگر استعفانامه‌ام را توی جیبم گذاشتم و بی این که صدایش را در بیاورم، روز سور هم نرفتم.

بعد دیدم این طور که نمی‌شود. گفتم بروم قضایا را برای رئیس فرهنگ بگویم. و رفتم. سلام و احوالپرسی نشستم. اما چه بگویم؟ بگویم چون نمی‌خواستم در خوردن سور شرکت کنم، استعفا می‌دهم؟... دیدم چیزی ندارم که بگویم. و از این گذشته خفت‌آور نبود که به خاطر سیصد تومان جا بزنم و استعفا بدهم؟ و «خداحافظ؛ فقط آمده بودم سلام عرض کنم.» و از این دروغ‌ها و استعفانامه‌ام را توی جوی آب انداختم. اما ناظم؛ یک هفته‌ای مثل سگ بود. عصبانی، پر سر و صدا و شارت و شورت! حتی نرفتم احوال مادرش را بپرسم. یک هفته‌ی تمام می‌رفتم و در اتاقم را می‌بستم و سوراخ‌های گوشم را می‌گرفتم و تا اِز و چِزّ بچه‌ها بخوابد، از این سر تا آن سر اتاق را می‌کوبیدم. ده روز تمام، قلب من و بچه‌ها با هم و به یک اندازه از ترس و وحشت تپید. تا عاقبت پول‌ها وصول شد. منتها به جای سیصد و خرده‌ای، فقط صد و پنجاه تومان. علت هم این بود که در تنظیم صورت حساب‌ها اشتباهاتی رخ داده بود که ناچار اصلاحش کرده بودند!

غیر از آن زنی که هفته‌ای یک بار به مدرسه سری می‌زد، از اولیای اطفال دو سه نفر دیگر هم بودند که مرتب بودند. یکی همان پاسبانی که با کمربند، پاهای پسرش را بست و فلک کرد. یکی هم کارمند پست و تلگرافی بود که ده روزی یک بار می‌آمد و پدر همان بچه‌ی شیطان. و یک استاد نجار که پسرش کلاس اول بود و خودش سواد داشت و به آن می‌بالید و کارآمد می‌نمود. یک مقنی هم بود درشت استخوان و بلندقد که بچه‌اش کلاس سوم بود و هفته‌ای یک بار می‌آمد و همان توی حیاط، ده پانزده دقیقه‌ای با فراش‌ها اختلاط می‌کرد و بی سر و صدا می‌رفت. نه کاری داشت، نه چیزی از آدم می‌خواست و همان طور که آمده بود چند دقیقه‌ای را با فراش صحبت می‌کرد و بعد می رفت. فقط یک روز نمی‌دانم چرا رفته بود بالای دیوار مدرسه. البته اول فکر کردم مأمور اداره برق است ولی بعد متوجه شدم که همان مرد مقنی است. بچه‌ها جیغ و فریاد می‌کردند و من همه‌اش درین فکر بودم که چه طور به سر دیوار رفته است؟ ماحصل داد و فریادش این بود که چرا اسم پسر او را برای گرفتن کفش و لباس به انجمن ندادیم. وقتی به او رسیدم نگاهی به او انداختم و بعد تشری به ناظم و معلم ها زدم که ولش کردند و بچه‌ها رفتند سر کلاس و بعد بی این که نگاهی به او بکنم، گفتم:

- خسته نباشی اوستا.

و همان طور که به طرف دفتر می‌رفتم رو به ناظم و معلم‌ها افزودم:

- لابد جواب درست و حسابی نشنیده که رفته سر دیوار.

که پشت سرم گرپ صدایی آمد و از در دفتر که رفتم تو، او و ناظم با هم وارد شدند. گفتم نشست. و به جای این‌که حرفی بزند به گریه افتاد. هرگز گمان نمی‌کردم از چنان قد و قامتی صدای گریه در بیاید. این بود که از اتاق بیرون آمدم و فراش را صدا زدم که آب برایش بیاورد و حالش که جا آمد، بیاوردش پهلوی من. اما دیگر از او خبری نشد که نشد. نه آن روز و نه هیچ روز دیگر. آن روز چند دقیقه‌ای بعد از شیشه‌ی اتاق خودم دیدمش که دمش را لای پایش گذاشته بود از در مدرسه بیرون می‌رفت و فراش جدید آمد که بله می‌گفتند از پسرش پنج تومان خواسته بودند تا اسمش را برای کفش و لباس به انجمن بدهند. پیدا بود باز توی کوک ناظم رفته است. مرخصش کردم و ناظم را خواستم. معلوم شد می‌خواسته ناظم را بزند. همین جوری و بی‌مقدمه.

اواخر بهمن بود که یکی از روزهای برفی با یکی دیگر از اولیای اطفال آشنا شدم. یارو مرد بسیار کوتاهی بود؛ فرنگ مآب و بزک کرده و اتو کشیده که ننشسته از تحصیلاتش و از سفرهای فرنگش حرف زد. می‌خواست پسرش را آن وقت سال از مدرسه‌ی دیگر به آن جا بیاورد. پسرش از آن بچه‌هایی بود که شیر و مربای صبحانه‌اش را با قربان صدقه توی حلقشان می‌تپانند. کلاس دوم بود و ثلث اول دو تا تجدید آورده بود. می‌گفت در باغ ییلاقی‌اش که نزدیک مدرسه است، باغبانی دارند که پسرش شاگرد ماست و درس‌خوان است و پیدا است که بچه‌ها زیر سایه شما خوب پیشرفت می‌کنند. و از این پیزرها. و حال به خاطر همین بچه، توی این برف و سرما، آمده‌اند ساکن باغ ییلاقی شده‌اند. بلند شدم ناظم را صدا کردم و دست او و بچه‌اش را توی دست ناظم گذاشتم و خداحافظ شما... و نیم ساعت بعد ناظم برگشت که یارو خانه‌ی شهرش را به یک دبیرستان اجاره داده، به ماهی سه هزار و دویست تومان، و التماس دعا داشته، یعنی معلم سرخانه می‌خواسته و حتی بدش نمی‌آمده است که خود مدیر زحمت بکشند و ازین گنده‌گوزی‌ها... احساس کردم که ناظم دهانش آب افتاده است. و من به ناظم حالی کردم خودش برود بهتر است و فقط کاری بکند که نه صدای معلم‌ها در بیاید و نه آخر سال، برای یک معدل ده احتیاجی به من بمیرم و تو بمیری پیدا کند. همان روز عصر ناظم رفته بود و قرار و مدار برای هر روز عصر یک ساعت به ماهی صد و پنجاه تومان.

دیگر دنیا به کام ناظم بود. حال مادرش هم بهتر بود و از بیمارستان مرخصش کرده بودند و به فکر زن گرفتن افتاده بود. و هر روز هم برای یک نفر نقشه می‌کشید حتی برای من هم. یک روز در آمد که چرا ما خودمان «انجمن خانه و مدرسه» نداشته باشیم؟ نشسته بود و حسابش را کرده بود دیده بود که پنجاه شصت نفری از اولیای مدرسه دستشان به دهان‌شان می‌رسد و از آن هم که به پسرش درس خصوصی می‌داد قول مساعد گرفته بود. حالیش کردم که مواظب حرف و سخن اداره‌ای باشد و هر کار دلش می‌خواهد بکند. کاغذ دعوت را هم برایش نوشتم با آب و تاب و خودش برای اداره‌ی فرهنگ، داد ماشین کردند و به وسیله‌ی خود بچه‌ها فرستاد. و جلسه با حضور بیست و چند نفری از اولیای بچه‌ها رسمی شد. خوبیش این بود که پاسبان کشیک پاسگاه هم آمده بود و دم در برای همه، پاشنه‌هایش را به هم می‌کوبید و معلم‌ها گوش تا گوش نشسته بودند و مجلس ابهتی داشت و ناظم، چای و شیرینی تهیه کرده بود و چراغ زنبوری کرایه کرده بود و باران هم گذاشت پشتش و سالون برای اولین بار در عمرش به نوایی رسید.

یک سرهنگ بود که رئیسش کردیم و آن زن را که هفته‌ای یک بار می‌آمد نایب رئیس. آن که ناظم به پسرش درس خصوصی می‌داد نیامده بود. اما پاکت سربسته‌ای به اسم مدیر فرستاده بود که فی‌المجلس بازش کردیم. عذرخواهی از این‌که نتوانسته بود بیاید و وجه ناقابلی جوف پاکت. صد و پنجاه تومان. و پول را روی میز صندوق‌دار گذاشتیم که ضبط و ربط کند. نائب رئیس بزک کرده و معطر شیرینی تعارف می‌کرد و معلم‌ها با هر بار که شیرینی بر می‌داشتند، یک بار تا بناگوش سرخ می‌شدند و فراش‌ها دست به دست چای می‌آوردند.

در فکر بودم که یک مرتبه احساس کردم، سیصد چهارصد تومان پول نقد، روی میز است و هشت صد تومان هم تعهد کرده بودند. پیرزن صندوقدار که کیف پولش را همراهش نیاورده بود ناچار حضار تصویب کردند که پول‌ها فعلاً پیش ناظم باشد. و صورت مجلس مرتب شد و امضاها ردیف پای آن و فردا فهمیدم که ناظم همان شب روی خشت نشسته بوده و به معلم‌ها سور داده بوده است. اولین کاری که کردم رونوشت مجلس آن شب را برای اداره‌ی فرهنگ فرستادم. و بعد همان استاد نجار را صدا کردم و دستور دادم برای مستراح‌ها دو روزه در بسازد که ناظم خیلی به سختی پولش را داد. و بعد در کوچه‌ی مدرسه درخت کاشتیم. تور والیبال را تعویض و تعدادی توپ در اختیار بچه‌ها گذاشتیم برای تمرین در بعد از ظهرها و آمادگی برای مسابقه با دیگر مدارس و در همین حین سر و کله‌ی بازرس تربیت بدنی هم پیدا شد و هر روز سرکشی و بیا و برو. تا یک روز که به مدرسه رسیدم شنیدم که از سالون سر و صدا می‌آید. صدای هالتر بود. ناظم سر خود رفته بود و سرخود دویست سیصد تومان داده بود و هالتر خریده بود و بچه‌های لاغر زیر بار آن گردن خود را خرد می‌کردند. من در این میان حرفی نزدم. می‌توانستم حرفی بزنم؟ من چیکاره بودم؟ اصلاً به من چه ربطی داشت؟ هر کار که دلشان می‌خواهد بکنند. مهم این بود که سالون مدرسه رونقی گرفته بود. ناظم هم راضی بود و معلم‌ها هم. چون نه خبر از حسادتی بود و نه حرف و سخنی پیش آمد. فقط می‌بایست به ناظم سفارش می کردم که فکر فراش‌ها هم باشد.

کم کم خودمان را برای امتحان‌های ثلث دوم آماده می‌کردیم. این بود که اوایل اسفند، یک روز معلم‌ها را صدا زدم و در شورا مانندی که کردیم بی‌مقدمه برایشان داستان یکی از همکاران سابقم را گفتم که هر وقت بیست می‌داد تا دو روز تب داشت. البته معلم‌ها خندیدند. ناچار تشویق شدم و داستان آخوندی را گفتم که در بچگی معلم شرعیاتمان بود و زیر عبایش نمره می‌داد و دستش چنان می‌لرزید که عبا تکان می‌خورد و درست ده دقیقه طول می‌کشید. و تازه چند؟ بهترین شاگردها دوازده. و البته باز هم خندیدند. که این بار کلافه‌ام کرد. و بعد حالیشان کردم که بد نیست در طرح سؤال‌ها مشورت کنیم و از این حرف‌ها...

و از شنبه‌ی بعد، امتحانات شروع شد. درست از نیمه‌ی دوم اسفند. سؤال‌ها را سه نفری می‌دیدیم. خودم با معلم هر کلاس و ناظم. در سالون میزها را چیده بودیم البته از وقتی هالتردار شده بود خیلی زیباتر شده بود. در سالون کاردستی‌های بچه‌ها در همه جا به چشم می‌خورد. هر کسی هر چیزی را به عنوان کاردستی درست کرده بودند و آورده بودند. که برای این کاردستی‌ها چه پول‌ها که خرج نشده بود و چه دست‌ها که نبریده بود و چه دعواها که نشده بود و چه عرق‌ها که ریخته نشده بود. پیش از هر امتحان که می‌شد، خودم یک میتینگ برای بچه‌ها می‌دادم که ترس از معلم و امتحان بی‌جا است و باید اعتماد به نفس داشت و ازین مزخرفات....ولی مگر حرف به گوش کسی می‌رفت؟ از در که وارد می‌شدند، چنان هجومی می‌بردند که نگو! به جاهای دور از نظر. یک بار چنان بود که احساس کردم مثل این‌که از ترس، لذت می‌برند. اگر معلم نبودی یا مدیر، به راحتی می‌توانستی حدس بزنی که کی‌ها با هم قرار و مداری دارند و کدام یک پهلو دست کدام یک خواهد نشست. یکی دو بار کوشیدم بالای دست یکی‌شان بایستم و ببینم چه می‌نویسد. ولی چنان مضطرب می‌شدند و دستشان به لرزه می‌افتاد که از نوشتن باز می‌ماندند. می‌دیدم که این مردان آینده، درین کلاس‌ها و امتحان‌ها آن قدر خواهند ترسید که وقتی دیپلمه بشوند یا لیسانسه، اصلاً آدم نوع جدیدی خواهند شد. آدمی انباشته از وحشت، انبانی از ترس و دلهره. به این ترتیب یک روز بیشتر دوام نیاوردم. چون دیدم نمی‌توانم قلب بچگانه‌ای داشته باشم تا با آن ترس و وحشت بچه‌ها را درک کنم و هم‌دردی نشان بدهم.این جور بود که می‌دیدم که معلم مدرسه هم نمی‌توانم باشم.

دو روز قبل از عید کارنامه‌ها آماده بود و منتظر امضای مدیر. دویست و سی و شش تا امضا اقلاً تا ظهر طول می‌کشید. پیش از آن هم تا می‌توانستم از امضای دفترهای حضور و غیاب می‌گریختم. خیلی از جیره‌خورهای دولت در ادارات دیگر یا در میان همکارانم دیده بودم که در مواقع بیکاری تمرین امضا می‌کنند. پیش از آن نمی‌توانستم بفهمم چه طور از مدیری یک مدرسه یا کارمندی ساده یک اداره می‌شود به وزارت رسید. یا اصلاً آرزویش را داشت. نیم‌قراضه امضای آماده و هر کدام معرف یک شخصیت، بعد نیم‌ذرع زبان چرب و نرم که با آن، مار را از سوراخ بیرون بکشی، یا همه جا را بلیسی و یک دست هم قیافه. نه یک جور. دوازده جور.

در این فکرها بودم که ناگهان در میان کارنامه‌ها چشمم به یک اسم آشنا افتاد. به اسم پسران جناب سرهنگ که رئیس انجمن بود. رفتم توی نخ نمراتش. همه متوسط بود و جای ایرادی نبود. و یک مرتبه به صرافت افتادم که از اول سال تا به حال بچه‌های مدرسه را فقط به اعتبار وضع مالی پدرشان قضاوت کرده‌ام. درست مثل این پسر سرهنگ که به اعتبار کیابیای پدرش درس نمی‌خواند. دیدم هر کدام که پدرشان فقیرتر است به نظر من باهوش‌تر می‌آمده‌اند. البته ناظم با این حرف‌ها کاری نداشت. مر قانونی را عمل می‌کرد. از یکی چشم می‌پوشید به دیگری سخت می‌گرفت.

اما من مثل این که قضاوتم را درباره‌ی بچه‌ها از پیش کرده باشم و چه خوب بود که نمره‌ها در اختیار من نبود و آن یکی هم «انظباط» مال آخر سال بود. مسخره‌ترین کارها آن است که کسی به اصلاح وضعی دست بزند، اما در قلمروی که تا سر دماغش بیشتر نیست. و تازه مدرسه‌ی من، این قلمروی فعالیت من، تا سر دماغم هم نبود. به همان توی ذهنم ختم می‌شد. وضعی را که دیگران ترتیب داده بودند. به این ترتیب بعد از پنج شش ماه، می‌فهمیدم که حسابم یک حساب عقلایی نبوده است. احساساتی بوده است. ضعف‌های احساساتی مرا خشونت‌های عملی ناظم جبران می‌کرد و این بود که جمعاً نمی‌توانستم ازو بگذرم. مرد عمل بود. کار را می‌برید و پیش می‌رفت. در زندگی و در هر کاری، هر قدمی بر می‌داشت، برایش هدف بود. و چشم از وجوه دیگر قضیه می‌پوشید. این بود که برش داشت. و من نمی‌توانستم. چرا که اصلاً مدیر نبودم. خلاص...

و کارنامه‌ی پسر سرهنگ را که زیر دستم عرق کرده بود، به دقت و احتیاج خشک کردم و امضایی زیر آن گذاشتم به قدری بد خط و مسخره بود که به یاد امضای فراش جدیدمان افتادم. حتماً جناب سرهنگ کلافه می‌شد که چرا چنین آدم بی‌سوادی را با این خط و ربط امضا مدیر مدرسه کرده‌اند. آخر یک جناب سرهنگ هم می‌داند که امضای آدم معرف شخصیت آدم است.

اواخر تعطیلات نوروز رفتم به ملاقات معلم ترکه‌ای کلاس سوم. ناظم که با او میانه‌ی خوشی نداشت. ناچار با معلم حساب کلاس پنج و شش قرار و مداری گذاشته بودم که مختصری علاقه‌ای هم به آن حرف و سخن‌ها داشت. هم به وسیله‌ی او بود که می‌دانستم نشانی‌اش کجا است و توی کدام زندان است. در راه قبل از هر چیز خبر داد که رئیس فرهنگ عوض شده و این طور که شایع است یکی از هم دوره‌ای‌های من، جایش آمده. گفتم:

- عجب! چرا؟ مگه رئیس قبلی چپش کم بود؟

- چه عرض کنم. می‌گند پا تو کفش یکی از نماینده‌ها کرده. شما خبر ندارید؟

- چه طور؟ از کجا خبر داشته باشم؟

- هیچ چی... می گند دو تا از کارچاق‌کن‌های انتخاباتی یارو از صندوق فرهنگ حقوق می‌گرفته‌اند؛ شب عیدی رئیس فرهنگ حقوق‌شون رو زده.

- عجب! پس اونم می‌خواسته اصلاحات کنه! بیچاره.

و بعد از این حرف زدیم که الحمدالله مدرسه مرتب است و آرام و معلم‌ها همکاری می‌کنند و ناظم بیش از اندازه همه‌کاره شده است. و من فهمیدم که باز لابد مشتری خصوصی تازه‌ای پیدا شده است که سر و صدای همه همکارها بلند شده. دم در زندان شلوغ بود. کلاه مخملی‌ها، عم‌قزی گل‌بته‌ها، خاله خانباجی‌ها و... اسم نوشتیم و نوبت گرفتیم و به جای پاها، دست‌هامان زیر بار کوچکی که داشتیم، خسته شد و خواب رفت تا نوبتمان شد. از این اتاق به آن اتاق و عاقبت نرده‌های آهنی و پشت آن معلم کلاس سه و... عجب چاق شده بود!درست مثل یک آدم حسابی شده بود. خوشحال شدیم و احوالپرسی و تشکر؛ و دیگر چه بگویم؟ بگویم چرا خودت را به دردسر انداختی؟ پیدا بود از مدرسه و کلاس به او خوش‌تر می‌گذرد. ایمانی بود و او آن را داشت و خوشبخت بود و دردسری نمی‌دید و زندان حداقل برایش کلاس درس بود. عاقبت پرسیدم:

- پرونده‌ای هم برات درست کردند یا هنوز بلاتکلیفی؟

- امتحانمو دادم آقا مدیر، بد از آب در نیومد.

- یعنی چه؟

- یعنی بی‌تکلیف نیستم. چون اسمم تو لیست جیره‌ی زندون رفته. خیالم راحته. چون سختی‌هاش گذشته.

دیگر چه بگویم. دیدم چیزی ندارم خداحافظی کردم و او را با معلم حساب تنها گذاشتم و آمدم بیرون و تا مدت ملاقات تمام بشود، دم در زندان قدم زدم و به زندانی فکر کردم که برای خودم ساخته بودم. یعنی آن خرپول فرهنگ‌دوست ساخته بود. و من به میل و رغبت خودم را در آن زندانی کرده بودم. این یکی را به ضرب دگنک این جا آورده بودند. ناچار حق داشت که خیالش راحت باشد. اما من به میل و رغبت رفته بودم و چه بکنم؟ ناظم چه طور؟ راستی اگر رئیس فرهنگ از هم دوره‌ای‌های خودم باشد؛ چه طور است بروم و ازو بخواهم که ناظم را جای من بگذارد، یا همین معلم حساب را؟... که معلم حساب در آمد و راه افتادیم. با او هم دیگر حرفی نداشتم. سر پیچ خداحافظ شما و تاکسی گرفتم و یک سر به اداره‌ی فرهنگ زدم. گرچه دهم عید بود، اما هنوز رفت و آمد سال نو تمام نشده بود. برو و بیا و شیرینی و چای دو جانبه. رفتم تو. سلام و تبریک و همین تعارفات را پراندم.

بله خودش بود. یکی از پخمه‌های کلاس. که آخر سال سوم کشتیارش شدم دو بیت شعر را حفظ کند، نتوانست که نتوانست. و حالا او رئیس بود و من آقا مدیر. راستی حیف از من، که حتی وزیر چنین رئیس فرهنگ‌هایی باشم! میز همان طور پاک بود و رفته. اما زیرسیگاری انباشته از خاکستر و ته سیگار. بلند شد و چلپ و چولوپ روبوسی کردیم و پهلوی خودش جا باز کرد و گوش تا گوش جیره‌خورهای فرهنگ تبریکات صمیمانه و بدگویی از ماسبق و هندوانه و پیزرها! و دو نفر که قد و قواره‌شان به درد گود زورخانه می‌خورد یا پای صندوق انتخابات شیرینی به مردم می‌دادند. نزدیک بود شیرینی را توی ظرفش بیندازم که دیدم بسیار احمقانه است. سیگارم که تمام شد قضیه‌ی رئیس فرهنگ قبلی و آن دو نفر را در گوشی ازش پرسیدم، حرفی نزد. فقط نگاهی می‌کرد که شبیه التماس بود و من فرصت جستم تا وضع معلم کلاس سوم را برایش روشن کنم و از او بخواهم تا آن جا که می‌تواند جلوی حقوقش را نگیرد. و از در که آمدم بیرون، تازه یادم آمد که برای کار دیگری پیش رئیس فرهنگ بودم.

باز دیروز افتضاحی به پا شد. معقول یک ماهه‌ی فروردین راحت بودیم. اول اردیبهشت ماه جلالی و کوس رسوایی سر دیوار مدرسه. نزدیک آخر وقت یک جفت پدر و مادر، بچه‌شان در میان، وارد اتاق شدند. یکی بر افروخته و دیگری رنگ و رو باخته و بچه‌شان عیناً مثل این عروسک‌های کوکی. سلام و علیک و نشستند. خدایا دیگر چه اتفاقی افتاده است؟

- چه خبر شده که با خانوم سرافرازمون کردید؟

مرد اشاره‌ای به زنش کرد که بلند شد و دست بچه را گرفت و رفت بیرون و من ماندم و پدر. اما حرف نمی‌زد. به خودش فرصت می‌داد تا عصبانیتش بپزد. سیگارم را در آوردم و تعارفش کردم. مثل این که مگس مزاحمی را از روی دماغش بپراند، سیگار را رد کرد و من که سیگارم را آتش می‌زدم، فکر کردم لابد دردی دارد که چنین دست و پا بسته و چنین متکی به خانواده به مدرسه آمده. باز پرسیدم:

- خوب، حالا چه فرمایش داشتید؟

که یک مرتبه ترکید:

- اگه من مدیر مدرسه بودم و هم‌چه اتفاقی می‌افتاد، شیکم خودمو پاره می‌کردم. خجالت بکش مرد! برو استعفا بده. تا اهل محل نریختن تیکه تیکه‌ات کنند، دو تا گوشتو وردار و دررو. بچه‌های مردم می‌آن این جا درس بخونن و حسن اخلاق. نمی‌آن که...

- این مزخرفات کدومه آقا! حرف حساب سرکار چیه؟

و حرکتی کردم که او را از در بیندازم بیرون. اما آخر باید می‌فهمیدم چه مرگش است. «ولی آخر با من چه کار دارد؟»

- آبروی من رفته. آبروی صد ساله‌ی خونواده‌ام رفته. اگه در مدرسه‌ی تو رو تخته نکنم، تخم بابام نیستم. آخه من دیگه با این بچه چی کار کنم؟ تو این مدرسه ناموس مردم در خطره. کلانتری فهمیده؛ پزشک قانونی فهمیده؛ یک پرونده درست شده پنجاه ورق؛ تازه می‌گی حرف حسابم چیه؟ حرف حسابم اینه که صندلی و این مقام از سر تو زیاده. حرف حسابم اینه که می‌دم محاکمه‌ات کنند و از نون خوردن بندازنت...

او می‌گفت و من گوش می‌کردم و مثل دو تا سگ هار به جان هم افتاده بودیم که در باز شد و ناظم آمد تو. به دادم رسید. در همان حال که من و پدر بچه در حال دعوا بودیم زن و بچه همان آقا رفته بودند و قضایا را برای ناظم تعریف کرده بودند و او فرستاده بوده فاعل را از کلاس کشیده بودند بیرون... و گفت چه طور است زنگ بزنیم و جلوی بچه‌ها ادبش کنیم و کردیم. یعنی این بار خود من رفتم میدان. پسرک نره‌خری بود از پنجمی‌ها با لباس مرتب و صورت سرخ و سفید و سالکی به گونه. جلوی روی بچه‌ها کشیدمش زیر مشت و لگد و بعد سه تا از ترکه‌ها را که فراش جدید فوری از باغ همسایه آورده بود، به سر و صورتش خرد کردم. چنان وحشی شده بودم که اگر ترکه‌ها نمی‌رسید، پسرک را کشته بودم. این هم بود که ناظم به دادش رسید و وساطت کرد و لاشه‌اش را توی دفتر بردند و بچه‌ها را مرخص کردند و من به اتاقم برگشتم و با حالی زار روی صندلی افتادم، نه از پدر خبری بود و نه از مادر و نه از عروسک‌های کوکی‌شان که ناموسش دست کاری شده بود. و تازه احساس کردم که این کتک‌کاری را باید به او می‌زدم. خیس عرق بودم و دهانم تلخ بود. تمام فحش‌هایی که می‌بایست به آن مردکه‌ی دبنگ می‌دادم و نداده بودم، در دهانم رسوب کرده بود و مثل دم مار تلخ شده بود. اصلاً چرا زدمش؟ چرا نگذاشتم مثل همیشه ناظم میدان‌داری کند که هم کارکشته‌تر بود و هم خونسردتر. لابد پسرک با دخترعمه‌اش هم نمی‌تواند بازی کند. لابد توی خانواده‌شان، دخترها سر ده دوازده سالگی باید از پسرهای هم سن رو بگیرند. نکند عیبی کرده باشد؟ و یک مرتبه به صرافت افتادم که بروم ببینم چه بلایی به سرش آورده‌ام. بلند شدم و یکی از فراش‌ها را صدا کردم که فهمیدم روانه‌اش کرده‌اند. آبی آورد که روی دستم می‌ریخت و صورتم را می‌شستم و می‌کوشیدم که لرزش دست‌هایم را نبیند. و در گوشم آهسته گفت که پسر مدیر شرکت اتوبوسرانی است و بدجوری کتک خورده و آن‌ها خیلی سعی کرده‌اند که تر و تمیزش کنند...

احمق مثلا داشت توی دل مرا خالی می‌کرد. نمی‌دانست که من اول تصمیم را گرفتم، بعد مثل سگ هار شدم. و تازه می‌فهمیدم کسی را زده‌ام که لیاقتش را داشته. حتماً از این اتفاق‌ها جای دیگر هم می‌افتد. آدم بردارد پایین تنه بچه‌ی خودش را، یا به قول خودش ناموسش را بگذارد سر گذر که کلانتر محل و پزشک معاینه کنند! تا پرونده درست کنند؟ با این پدرو مادرها بچه‌ها حق دارند که قرتی و دزد و دروغگو از آب در بیایند. این مدرسه‌ها را اول برای پدر و مادرها باز کنند...

با این افکار به خانه رسیدم. زنم در را که باز کرد؛ چشم‌هایش گرد شد. همیشه وقتی می‌ترسد این طور می‌شود. برای اینکه خیال نکند آدم کشته‌ام، زود قضایا را برایش گفتم. و دیدم که در ماند. یعنی ساکت ماند. آب سرد، عرق بیدمشک، سیگار پشت سیگار فایده نداشت، لقمه از گلویم پایین نمی‌رفت و دست‌ها هنوز می‌لرزید. هر کدام به اندازه‌ی یک ماه فعالیت کرده بودند. با سیگار چهارم شروع کردم:

- می‌دانی زن؟ بابای یارو پول‌داره. مسلماً کار به دادگستری و این جور خنس‌ها می‌کشه. مدیریت که الفاتحه. اما خیلی دلم می‌خواد قضیه به دادگاه برسه. یک سال آزگار رو دل کشیده‌ام و دیگه خسته شده‌ام. دلم می‌خواد یکی بپرسه چرا بچه‌ی مردم رو این طوری زدی، چرا تنبیه بدنی کردی! آخه یک مدیر مدرسه هم حرف‌هایی داره که باید یک جایی بزنه...

که بلند شد و رفت سراغ تلفن. دو سه تا از دوستانم را که در دادگستری کاره‌ای بودند، گرفت و خودم قضیه را برایشان گفتم که مواظب باشند. فردا پسرک فاعل به مدرسه نیامده بود. و ناظم برایم گفت که قضیه ازین قرار بوده است که دوتایی به هوای دیدن مجموعه تمبرهای فاعل با هم به خانه‌ای می‌روند و قضایا همان جا اتفاق می‌افتد و داد و هوار و دخالت پدر و مادرهای طرفین و خط و نشان و شبانه کلانتری؛ و تمام اهل محل خبر دارند. او هم نظرش این بود که کار به دادگستری خواهد کشید.

و من یک هفته‌ی تمام به انتظار اخطاریه‌ی دادگستری صبح و عصر به مدرسه رفتم و مثل بخت‌النصر پشت پنجره ایستادم. اما در تمام این مدت نه از فاعل خبری شد، نه از مفعول و نه از پدر و مادر ناموس‌پرست و نه از مدیر شرکت اتوبوسرانی. انگار نه انگار که اتفاقی افتاده. بچه‌ها می‌آمدند و می‌رفتند؛ برای آب خوردن عجله می‌کردند؛ به جای بازی کتک‌کاری می‌کردند و همه چیز مثل قبل بود. فقط من ماندم و یک دنیا حرف و انتظار. تا عاقبت رسید.... احضاریه‌ای با تعیین وقت قبلی برای دو روز بعد، در فلان شعبه و پیش فلان بازپرس دادگستری. آخر کسی پیدا شده بود که به حرفم گوش کند.

تا دو روز بعد که موعد احضار بود، اصلاً از خانه در نیامدم. نشستم و ماحصل حرف‌هایم را روی کاغذ آوردم. حرف‌هایی که با همه‌ی چرندی هر وزیر فرهنگی می‌توانست با آن یک برنامه‌ی هفت ساله برای کارش بریزد. و سر ساعت معین رفتم دادگستری. اتاق معین و بازپرس معین. در را باز کردم و سلام، و تا آمدم خودم را معرفی کنم و احضاریه را در بیاورم، یارو پیش‌دستی کرد و صندلی آورد و چای سفارش داد و «احتیاجی به این حرف‌ها نیست و قضیه‌ی کوچک بود و حل شد و راضی به زحمت شما نبودیم...»

که عرق سرد بر بدن من نشست. چایی‌ام را که خوردم، روی همان کاغذ نشان‌دار دادگستری استعفانامه‌ام را نوشتم و به نام هم‌کلاسی پخمه‌ام که تازه رئیس شده بود، دم در پست کردم.
EOT;
}
